/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.guarded.GuardedAsyncProcedureWrapper;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.impl.StaticObjectAdapter;
import org.simantics.g2d.elementclass.connection.ConnectionClass;
import org.simantics.g2d.routing.IRouter2;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * An element class for single connection entity elements. A connection entity
 * consists of connection edge segments and branch points as its children.
 * 
 * @author Tuukka Lehtonen
 */
public class DefaultConnectionClassFactory extends ElementFactoryAdapter {

    public static final ElementClass CLASS = ConnectionClass.CLASS;

    @Override
    public void create(AsyncReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType, final AsyncProcedure<ElementClass> procedure) {
        procedure.execute(graph, ConnectionClass.CLASS.newClassWith(false, new StaticObjectAdapter(elementType)));
    }

    @Override
    protected Resource getElementClassBaseType(AsyncReadGraph graph) {
        return graph.getService(DiagramResource.class).Connection;
    }

    @Override
    public void load(AsyncReadGraph graph, ICanvasContext canvas, IDiagram diagram, final Resource elementResource,
            final IElement element, final AsyncProcedure<IElement> procedure) {
        final GuardedAsyncProcedureWrapper<IElement> guard = new GuardedAsyncProcedureWrapper<IElement>(procedure, 1);

        // Get custom routing algorithm for connection if necessary.
        DiagramGraphUtil.getPossibleRouter(graph, elementResource, new AsyncProcedure<IRouter2>() {
            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                guard.exception(graph, throwable);
            }
            @Override
            public void execute(AsyncReadGraph graph, IRouter2 router) {
                if (router != null) {
                    element.setHint(DiagramHints.ROUTE_ALGORITHM, router);
                    procedure.execute(graph, element);
                } else {
                    loadConnectionTypeDefaultRouting(graph, element, elementResource, guard);
                }
            }
        });
    }

    protected void loadConnectionTypeDefaultRouting(AsyncReadGraph graph, final IElement element, Resource elementResource, final AsyncProcedure<IElement> guard) {
        StructuralResource2 STR = graph.getService(StructuralResource2.class);
        graph.forPossibleObject(elementResource, STR.HasConnectionType, new AsyncProcedure<Resource>() {
            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                guard.exception(graph, throwable);
            }
            @Override
            public void execute(AsyncReadGraph graph, Resource connectionType) {
                DiagramResource DIA = graph.getService(DiagramResource.class);
                if (connectionType != null) {
                    graph.forPossibleObject(connectionType, DIA.HasDefaultRouting, new AsyncProcedure<Resource>() {
                        @Override
                        public void exception(AsyncReadGraph graph, Throwable throwable) {
                            guard.exception(graph, throwable);
                        }
                        @Override
                        public void execute(AsyncReadGraph graph, Resource routing) {
                            if (routing != null) {
                                graph.forPossibleAdapted(routing, IRouter2.class, new AsyncProcedure<IRouter2>() {
                                    @Override
                                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                                        guard.exception(graph, throwable);
                                    }
                                    @Override
                                    public void execute(AsyncReadGraph graph, IRouter2 router) {
                                        if (router != null)
                                            element.setHint(DiagramHints.ROUTE_ALGORITHM, router);
                                        guard.execute(graph, element);
                                    }
                                });
                            } else {
                                guard.execute(graph, element);
                            }
                        }
                    });
                } else {
                    guard.execute(graph, element);
                }
            }
        });
    }

}
