/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.awt.Point;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.Bindings;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.procedure.guarded.GuardedAsyncProcedureWrapper;
import org.simantics.db.common.request.BinaryAsyncRead;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.elements.AnimatedSVGImage;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.TransformSynchronizer;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.element.handler.impl.SimpleElementLayers;
import org.simantics.g2d.element.handler.impl.StaticObjectAdapter;
import org.simantics.g2d.element.handler.impl.StaticSymbolImageInitializer;
import org.simantics.g2d.element.handler.impl.StaticSymbolImpl;
import org.simantics.g2d.element.handler.impl.TextImpl;
import org.simantics.g2d.elementclass.ImageClass.ImageElementHandler;
import org.simantics.g2d.image.DefaultImages;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.image.ProviderUtils;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.cache.IFactory;
import org.simantics.utils.datastructures.cache.ProvisionException;
import org.simantics.utils.strings.StringUtils;
import org.simantics.utils.ui.ErrorLogger;

/**
 * 
 */
public class AnimatedSVGElementClassFactory extends ElementFactoryAdapter {

    public static final ElementFactory INSTANCE = new AnimatedSVGElementClassFactory();

    private static final StaticSymbolImpl STATIC_SYMBOL = new StaticSymbolImpl(DefaultImages.SVG.get());

    static class ClassRequest extends BinaryAsyncRead<Resource, ICanvasContext, ElementClass> {

        final private IDiagram diagram;

        public ClassRequest(Resource elementType, ICanvasContext canvas, IDiagram diagram) {
            super(elementType, canvas);
            this.diagram = diagram;
        }

        @Override
        public void perform(AsyncReadGraph graph, AsyncProcedure<ElementClass> procedure) {
            createClass(graph, parameter2, diagram, parameter, procedure);
        }

    }

    @Override
    public void create(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram,
            final Resource elementType, final AsyncProcedure<ElementClass> procedure) {

        graph.asyncRequest(new ClassRequest(elementType, canvas, diagram), new TransientCacheAsyncListener<ElementClass>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                t.printStackTrace();
                procedure.exception(graph, t);
            }

            @Override
            public void execute(AsyncReadGraph graph, ElementClass result) {
                procedure.execute(graph, result);
            }

        });

    }

    public static void createClass(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram,
            final Resource elementType, final AsyncProcedure<ElementClass> procedure) {
        String id = "AnimatedSVGElement: " + elementType.getResourceId();
        //System.out.println(id);
        procedure.execute(graph, ElementClass.compile(
                TextImpl.INSTANCE,
                new StaticObjectAdapter(elementType),
                DefaultTransform.INSTANCE,
                StaticSymbolImageInitializer.INSTANCE,
                STATIC_SYMBOL,
                ImageElementHandler.INSTANCE,
                SimpleElementLayers.INSTANCE)
                .setId(id));
    }

    @Override
    public void load(AsyncReadGraph graph, final ICanvasContext canvas, final IDiagram diagram, final Resource element, final IElement e, final AsyncProcedure<IElement> procedure) {
        e.setHint(SynchronizationHints.HINT_SYNCHRONIZER, TransformSynchronizer.INSTANCE);

        final GuardedAsyncProcedureWrapper<IElement> guard = new GuardedAsyncProcedureWrapper<IElement>(procedure, 3);

        loadSVG(graph, diagram, element, e, guard);
        ElementFactoryUtil.loadLayersForElement(graph, diagram, e, element, guard);
        ElementFactoryUtil.readTransform(graph, element, e, guard);
    }

    private void loadSVG(AsyncReadGraph graph, final IDiagram diagram, final Resource element, final IElement e, final AsyncProcedure<IElement> guard) {
        final Layer0 L0 = graph.getService(Layer0.class);
        final G2DResource G2D = graph.getService(G2DResource.class);
        final Binding stringBinding = Bindings.STRING;

        graph.forPossibleRelatedValue(element, L0.HasName, stringBinding, new AsyncProcedure<String>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                guard.exception(graph, throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, final String name) {

                graph.forPossibleRelatedValue(element, G2D.HasSVGDocument, stringBinding, new AsyncProcedure<String>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        guard.exception(graph, throwable);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, final String svgDocument) {

                        graph.forPossibleRelatedValue(element, G2D.HasSVGScript, stringBinding, new AsyncProcedure<String>() {

                            @Override
                            public void exception(AsyncReadGraph graph, Throwable throwable) {
                                guard.exception(graph, throwable);
                            }

                            @Override
                            public void execute(AsyncReadGraph graph, final String svgScript) {
                                if (svgDocument != null) {
                                    Point referenceSize = diagram.getHint(DiagramHints.KEY_ELEMENT_RASTER_TARGET_SIZE);
                                    IFactory<Image> img = AnimatedSVGImage.createFactoryFromString(
                                            StringUtils.safeString(name) + String.valueOf(element.getResourceId()),
                                            svgDocument,
                                            StringUtils.safeString(svgScript),
                                            referenceSize);
                                    try {
                                        img = ProviderUtils.rasterize(img);
                                        e.setHint(ElementHints.KEY_IMAGE, img.get());
                                        guard.execute(graph, e);
                                    } catch (ProvisionException ex) {
                                        ErrorLogger.defaultLogError(ex);
                                        e.setHint(ElementHints.KEY_IMAGE, DefaultImages.SVG.get());
                                        guard.execute(graph, e);
                                    }
                                } else {
                                    e.setHint(ElementHints.KEY_IMAGE, DefaultImages.UNKNOWN.get());
                                    guard.execute(graph, e);
                                }
                            }
                        });
                    }
                });
            }
        });
    }

}