/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #1070
 *******************************************************************************/
package org.simantics.diagram;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead2;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.db.service.CollectionSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.symbolcontribution.CompositeSymbolProviderFactory;
import org.simantics.diagram.symbolcontribution.ISymbolProvider;
import org.simantics.diagram.symbolcontribution.SymbolContribution;
import org.simantics.diagram.symbolcontribution.SymbolProviderFactory;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.diagram.symbollibrary.ISymbolItem;
import org.simantics.utils.ui.AdaptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 */
public final class DiagramTypeUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(DiagramTypeUtils.class);

    public static SymbolProviderFactory readSymbolProviderFactory(RequestProcessor processor, final Resource diagram) throws DatabaseException {
        return processor.syncRequest(new Read<SymbolProviderFactory>() {
            @Override
            public SymbolProviderFactory perform(ReadGraph graph) throws DatabaseException {
                return readSymbolProviderFactory(graph, diagram);
            }
        });
    }

    public static SymbolProviderFactory readSymbolProviderFactory(ReadGraph graph, Resource diagram) throws DatabaseException {
        DiagramResource dr = DiagramResource.getInstance(graph);

        Collection<SymbolProviderFactory> factories = new ArrayList<SymbolProviderFactory>();
        CollectionSupport cs = graph.getService(CollectionSupport.class);

        for (Resource diagramType : cs.asSortedList(graph.getTypes(diagram))) {
            for (Resource symbolContribution : cs.asSortedList(graph.getObjects(diagramType, dr.HasSymbolContribution))) {
                try {
                    SymbolContribution contribution = graph.adaptUnique(symbolContribution, SymbolContribution.class);
                    factories.add(contribution.create(graph, diagram));
                } catch (DatabaseException e) {
                    LOGGER.error("Failed to adapt {} to SymbolContribution. Some symbols might be missing.", NameUtils.getURIOrSafeNameInternal(graph, symbolContribution), e); 
                }
            }
        }

        return new CompositeSymbolProviderFactory(factories);
    }

    public static ISymbolProvider readSymbolContributions(RequestProcessor processor, final Resource diagram) throws DatabaseException {
        return processor.syncRequest(new Read<ISymbolProvider>() {
            @Override
            public ISymbolProvider perform(ReadGraph graph) throws DatabaseException {
                return readSymbolContributions(graph, diagram);
            }
        });
    }

    public static ISymbolProvider readSymbolContributions(ReadGraph graph, Resource diagram) throws DatabaseException {
        return readSymbolProviderFactory(graph, diagram).create(graph);
    }


    public static boolean symbolAllowedOnDiagram(RequestProcessor processor, ISymbolItem symbol, final Resource diagram) throws DatabaseException {
        return symbolFromGroupAllowedOnDiagram(processor, symbol.getGroup(), diagram);
    }

    public static boolean symbolFromGroupAllowedOnDiagram(RequestProcessor processor, ISymbolGroup group, final Resource diagram) throws DatabaseException {
        final Resource groupResource = AdaptionUtils.adaptToSingle(group, Resource.class);
        return symbolFromGroupAllowedOnDiagram(processor, groupResource, diagram);
    }

    public static boolean symbolFromGroupAllowedOnDiagram(RequestProcessor processor, final Resource symbolGroup, final Resource diagram) throws DatabaseException {
        // no semantic info, no validation
        if (diagram == null)
            return true;
        if (symbolGroup == null)
            return true;

        return processor.syncRequest(new ResourceRead2<Boolean>(diagram, symbolGroup) {
            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                DiagramResource DIA = DiagramResource.getInstance(graph);
                for (Resource diagramType : graph.getTypes(diagram)) {
                    for (Resource symbolContribution : graph.getObjects(diagramType, DIA.HasSymbolContribution)) {
                        for (Resource symbolLibrary : graph.getObjects(symbolContribution, DIA.BasicSymbolContributionHasSymbolLibrary)) {
                            if (symbolGroup.equals(symbolLibrary))
                                return Boolean.TRUE;
                        }
                    }
                }
                return Boolean.FALSE;
            }
        });
    }

}
