/*******************************************************************************
 * Copyright (c) 2007, 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - adapter cache optimization
 *******************************************************************************/
package org.simantics.db.services.adaption;

import java.io.Serializable;

/**
 * Serializable representation of an adapter parameter definition with pre-resolved resource IDs.
 * This class is used in the adapter cache to avoid URI resolution during startup.
 * 
 * @author Tuukka Lehtonen
 * @since 1.67.0
 */
public class CachedParameterDefinition implements Serializable {
    
    private static final long serialVersionUID = 1L;
    
    /**
     * Type of this parameter
     */
    public ParameterType type;
    
    /**
     * Resource ID for relation-based parameters (RELATED, SINGLE, ORDERED_SET, AT_MOST_ONE).
     * Contains the resource ID of the relation to follow.
     */
    public long relationResourceId;
    
    /**
     * String constant value for STRING parameter type
     */
    public String constantValue;
    
    /**
     * Bundle symbolic name for BUNDLE parameter type
     */
    public String bundleId;
    
    /**
     * Target class name for adaptation (when "to" attribute is specified)
     */
    public String adaptTo;
    
    /**
     * Default constructor for serialization
     */
    public CachedParameterDefinition() {
    }
    
    /**
     * Constructor for simple parameter types (THIS, GRAPH)
     */
    public CachedParameterDefinition(ParameterType type) {
        this.type = type;
    }
    
    /**
     * Constructor for relation-based parameters
     */
    public CachedParameterDefinition(ParameterType type, long relationResourceId) {
        this.type = type;
        this.relationResourceId = relationResourceId;
    }
    
    /**
     * Constructor for string constant parameters
     */
    public CachedParameterDefinition(ParameterType type, String constantValue) {
        this.type = type;
        this.constantValue = constantValue;
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("CachedParameterDefinition[type=").append(type);
        if (relationResourceId != 0) {
            sb.append(", relationResourceId=").append(relationResourceId);
        }
        if (constantValue != null) {
            sb.append(", constantValue=").append(constantValue);
        }
        if (bundleId != null) {
            sb.append(", bundleId=").append(bundleId);
        }
        if (adaptTo != null) {
            sb.append(", adaptTo=").append(adaptTo);
        }
        sb.append("]");
        return sb.toString();
    }
}
