/*******************************************************************************
 * Copyright (c) 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.services.adaption;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Serializable representation of an adapter definition with pre-resolved resource IDs.
 * This class is used in the adapter cache to avoid XML parsing and URI resolution during startup.
 * 
 * @author Tuukka Lehtonen
 * @since 1.67.0
 */
public class CachedAdapterDefinition implements Serializable {
    
    private static final long serialVersionUID = 1L;
    
    /**
     * Type of adapter definition
     */
    public AdapterType type;
    
    /**
     * Resource ID of the type or resource this adapter is registered for.
     * For TYPE and BASE_TYPE: the type resource ID
     * For RESOURCE: the specific resource instance ID
     * For ADAPTER_CLASS: the type resource ID
     */
    public long typeResourceId;
    
    /**
     * Fully qualified class name of the target interface
     */
    public String targetInterface;
    
    /**
     * Fully qualified class name of the adapter implementation
     */
    public String adapterClass;
    
    /**
     * Fully qualified class name of the context class (optional)
     */
    public String contextClass;
    
    /**
     * Constructor method name for static factory methods (optional)
     */
    public String constructor;
    
    /**
     * List of parameters for adapter construction
     */
    public List<CachedParameterDefinition> parameters;
    
    /**
     * Symbolic name of the bundle providing this adapter
     */
    public String bundleSymbolicName;
    
    /**
     * Class name of custom AdapterInstaller implementation (for INSTALLER type)
     */
    public String installerClassName;
    
    /**
     * Source file path for debugging and logging
     */
    public String sourceFile;
    
    /**
     * Default constructor for serialization
     */
    public CachedAdapterDefinition() {
        this.parameters = new ArrayList<>();
    }
    
    /**
     * Constructor for base type declarations
     */
    public static CachedAdapterDefinition forBaseType(long typeResourceId, String targetInterface, 
            String bundleSymbolicName, String sourceFile) {
        CachedAdapterDefinition def = new CachedAdapterDefinition();
        def.type = AdapterType.BASE_TYPE;
        def.typeResourceId = typeResourceId;
        def.targetInterface = targetInterface;
        def.bundleSymbolicName = bundleSymbolicName;
        def.sourceFile = sourceFile;
        return def;
    }
    
    /**
     * Constructor for type adapters
     */
    public static CachedAdapterDefinition forType(long typeResourceId, String targetInterface,
            String adapterClass, String constructor, List<CachedParameterDefinition> parameters,
            String bundleSymbolicName, String sourceFile) {
        CachedAdapterDefinition def = new CachedAdapterDefinition();
        def.type = AdapterType.TYPE;
        def.typeResourceId = typeResourceId;
        def.targetInterface = targetInterface;
        def.adapterClass = adapterClass;
        def.constructor = constructor;
        def.parameters = parameters != null ? parameters : new ArrayList<>();
        def.bundleSymbolicName = bundleSymbolicName;
        def.sourceFile = sourceFile;
        return def;
    }
    
    /**
     * Constructor for resource instance adapters
     */
    public static CachedAdapterDefinition forResource(long resourceId, String targetInterface,
            String adapterClass, String constructor, List<CachedParameterDefinition> parameters,
            String bundleSymbolicName, String sourceFile) {
        CachedAdapterDefinition def = new CachedAdapterDefinition();
        def.type = AdapterType.RESOURCE;
        def.typeResourceId = resourceId;
        def.targetInterface = targetInterface;
        def.adapterClass = adapterClass;
        def.constructor = constructor;
        def.parameters = parameters != null ? parameters : new ArrayList<>();
        def.bundleSymbolicName = bundleSymbolicName;
        def.sourceFile = sourceFile;
        return def;
    }
    
    /**
     * Constructor for adapter class references
     */
    public static CachedAdapterDefinition forAdapterClass(long typeResourceId, String targetInterface,
            String adapterClass, String contextClass, String bundleSymbolicName, String sourceFile) {
        CachedAdapterDefinition def = new CachedAdapterDefinition();
        def.type = AdapterType.ADAPTER_CLASS;
        def.typeResourceId = typeResourceId;
        def.targetInterface = targetInterface;
        def.adapterClass = adapterClass;
        def.contextClass = contextClass;
        def.bundleSymbolicName = bundleSymbolicName;
        def.sourceFile = sourceFile;
        return def;
    }
    
    /**
     * Constructor for custom installer
     */
    public static CachedAdapterDefinition forInstaller(String installerClassName,
            String bundleSymbolicName, String sourceFile) {
        CachedAdapterDefinition def = new CachedAdapterDefinition();
        def.type = AdapterType.INSTALLER;
        def.installerClassName = installerClassName;
        def.bundleSymbolicName = bundleSymbolicName;
        def.sourceFile = sourceFile;
        return def;
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("CachedAdapterDefinition[type=").append(type);
        sb.append(", targetInterface=").append(targetInterface);
        sb.append(", adapterClass=").append(adapterClass);
        if (typeResourceId != 0) {
            sb.append(", typeResourceId=").append(typeResourceId);
        }
        if (contextClass != null) {
            sb.append(", contextClass=").append(contextClass);
        }
        if (constructor != null) {
            sb.append(", constructor=").append(constructor);
        }
        if (installerClassName != null) {
            sb.append(", installerClassName=").append(installerClassName);
        }
        sb.append(", bundle=").append(bundleSymbolicName);
        sb.append(", source=").append(sourceFile);
        sb.append("]");
        return sb.toString();
    }
}
