/*******************************************************************************
 * Copyright (c) 2007, 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - adapter cache optimization
 *******************************************************************************/
package org.simantics.db.services.adaption;

import java.io.File;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.adaption.Adapter;
import org.simantics.db.adaption.AdapterInstaller;
import org.simantics.db.adaption.AdaptionService;
import org.simantics.db.common.primitiverequest.PossibleResource;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.db.services.adaption.reflection.AdaptingDynamicAdapter2;
import org.simantics.db.services.adaption.reflection.AtMostOneRelatedResource2;
import org.simantics.db.services.adaption.reflection.ConstantAdapter;
import org.simantics.db.services.adaption.reflection.GraphObject2;
import org.simantics.db.services.adaption.reflection.IDynamicAdapter2;
import org.simantics.db.services.adaption.reflection.OrderedSetResources2;
import org.simantics.db.services.adaption.reflection.ReflectionAdapter2;
import org.simantics.db.services.adaption.reflection.RelatedResources2;
import org.simantics.db.services.adaption.reflection.SingleRelatedResource2;
import org.simantics.db.services.adaption.reflection.StaticMethodAdapter;
import org.simantics.db.services.adaption.reflection.ThisResource2;
import org.simantics.scl.reflection.OntologyVersions;
import org.simantics.utils.FileUtils;
import org.simantics.utils.threads.ThreadUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public class AdapterRegistry2 {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdapterRegistry2.class);

    public static final String ADAPTERS_FILE = "adapters.xml";

    public static final String ADAPTERS = "adapters";
    public static final String ADAPTER = "adapter";
    public static final String TARGET = "target";
    public static final String BASE_TYPE = "baseType";
    public static final String TYPE = "type";
    public static final String RESOURCE = "resource";
    public static final String URI = "uri";
    public static final String INTERFACE = "interface";
    public static final String CLASS = "class";
    public static final String ADAPTER_CLASS = "adapterClass";
    public static final String CONTEXT_CLASS = "contextClass";
    public static final String INSTALLER = "installer";
    public static final String CONSTRUCTOR = "constructor";

    ConcurrentHashMap<AdapterInstaller, String> installerSources = new ConcurrentHashMap<>();
    Collection<Exception> exceptions = new ArrayList<Exception>();
    List<CachedAdapterDefinition> cacheDefinitions = new ArrayList<>();
    boolean buildingCache = false;

    private void addInstaller(AdapterInstaller installer, String sourceDesc) {
        installerSources.put(installer, sourceDesc);
    }

    private static void handleException(Exception e, String fileName) {
        LOGGER.error("At {}", fileName, e);
    }

    private void handleException(Throwable e, AdapterInstaller installer) {
        String desc = installerSources.get(installer);
        LOGGER.error("At {}, installer {}", desc, installer, e);
    }

    private void handleAdaptersDocument(Loader b, Document doc, String fileName) {
        try {
            Node node = doc.getDocumentElement();
            if(node.getNodeName().equals(ADAPTERS)) {
                NodeList nodeList = node.getChildNodes();
                for(int i=0;i<nodeList.getLength();++i) {
                    Node n = nodeList.item(i);
                    if(n.getNodeName().equals(TARGET))
                        handleTarget(b, n, fileName);
                    else if(n.getNodeName().equals(INSTALLER))
                        handleInstaller(b, n, fileName);
                }
            }
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    private void handleTarget(Loader b, Node node, String fileName) {
        try {
            Class<?> interface_ =
                b.loadClass(node.getAttributes().getNamedItem("interface")
                        .getNodeValue());
            NodeList nodeList = node.getChildNodes();
            for(int i=0;i<nodeList.getLength();++i) {
                Node n = nodeList.item(i);
                String nodeName = n.getNodeName();
                if(nodeName.equals(BASE_TYPE))
                    handleBaseType(b, interface_, n, fileName);
                else if(nodeName.equals(TYPE))
                    handleType(b, interface_, n, fileName);
                else if(nodeName.equals(ADAPTER))
                    handleAdapter(b, interface_, n, fileName);
                else if(nodeName.equals(RESOURCE))
                    handleResource(b, interface_, n, fileName);
            }
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    private void handleInstaller(Loader b, Node node, String fileName) {
        try {
            final String installerClassName = node.getAttributes().getNamedItem("class").getNodeValue();
            AdapterInstaller installer =
                ((Class<?>)b.loadClass(installerClassName))
                .asSubclass(AdapterInstaller.class).getDeclaredConstructor().newInstance();
            addInstaller(installer, fileName);
            
            // Build cache definition for custom installers if in cache building mode
            if (buildingCache) {
                CachedAdapterDefinition def = CachedAdapterDefinition.forInstaller(
                        installerClassName,
                        b.getBundle() != null ? b.getBundle().getSymbolicName() : "unknown",
                        fileName);
                synchronized (cacheDefinitions) {
                    cacheDefinitions.add(def);
                }
            }
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    private <T> void handleResource(final Loader b, final Class<T> interface_, final Node node, String fileName) {
        try {
            NamedNodeMap attr = node.getAttributes();
            final String uri = attr.getNamedItem(URI).getNodeValue();
            final String className = attr.getNamedItem(CLASS).getNodeValue();
            Node constructorNode = attr.getNamedItem(CONSTRUCTOR);
            final String constructor = constructorNode == null ? null : constructorNode.getNodeValue();
//            System.out.println("AdapterRegistry2.handleResource: " + b + " " + uri + " " + interface_);
            addInstaller(

                    new AdapterInstaller() {

                        @Override
                        public void install(ReadGraph g, AdaptionService service) throws Exception {
                            Class<? extends T> clazz = b.loadClass(className).asSubclass(interface_);
                            List<IDynamicAdapter2> parameters = readParameters(g, node, b);
                            IDynamicAdapter2[] parameterArray = 
                                parameters.toArray(new IDynamicAdapter2[parameters.size()]);
                            Resource r = g.syncRequest(new PossibleResource(uri), TransientCacheAsyncListener.instance());
                            service.addInstanceAdapter(
                                    r,
                                    interface_,
                                    constructor == null 
                                    ? new ReflectionAdapter2<T>(clazz, parameterArray)
                                    : new StaticMethodAdapter<T>(clazz, constructor, parameterArray));
                            
                            // Build cache definition if in cache building mode
                            if (buildingCache && r != null) {
                                List<CachedParameterDefinition> cachedParams = buildCachedParameters(g, node, b);
                                CachedAdapterDefinition def = CachedAdapterDefinition.forResource(
                                        r.getResourceId(),
                                        interface_.getName(),
                                        className,
                                        constructor,
                                        cachedParams,
                                        b.getBundle() != null ? b.getBundle().getSymbolicName() : "unknown",
                                        fileName);
                                synchronized (cacheDefinitions) {
                                    cacheDefinitions.add(def);
                                }
                            }
                        }

                    }, fileName);
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    private <T> void handleType(final Loader b, final Class<T> interface_, final Node node, String fileName) {
        try {
            final NamedNodeMap attr = node.getAttributes();
            final String uri = attr.getNamedItem(URI).getNodeValue();
            Node constructorNode = attr.getNamedItem(CONSTRUCTOR);
            final String constructor = constructorNode == null ? null : constructorNode.getNodeValue();
            final String className = attr.getNamedItem(CLASS).getNodeValue();
            
            //System.out.println("AdapterRegistry2.handleType: " + b + " " + uri + " " + interface_);
            addInstaller(
                    new AdapterInstaller() {

                        @Override
                        public void install(ReadGraph g, AdaptionService service) throws Exception {
                            try {
                                Class<? extends T> clazz = ((Class<?>)b.loadClass(className)).asSubclass(interface_);
                                List<IDynamicAdapter2> parameters = readParameters(g, node, b);
                                IDynamicAdapter2[] parameterArray = 
                                    parameters.toArray(new IDynamicAdapter2[parameters.size()]);
                                Resource typeResource = g.getResource(uri);
                                service.addAdapter(
                                        typeResource,
                                        interface_,
                                        constructor == null 
                                        ? new ReflectionAdapter2<T>(clazz, parameterArray)
                                        : new StaticMethodAdapter<T>(clazz, constructor, parameterArray));

                                // Build cache definition if in cache building mode
                                if (buildingCache) {
                                    List<CachedParameterDefinition> cachedParams = buildCachedParameters(g, node, b);
                                    CachedAdapterDefinition def = CachedAdapterDefinition.forType(
                                            typeResource.getResourceId(),
                                            interface_.getName(),
                                            className,
                                            constructor,
                                            cachedParams,
                                            b.getBundle() != null ? b.getBundle().getSymbolicName() : "unknown",
                                            fileName);
                                    synchronized (cacheDefinitions) {
                                        cacheDefinitions.add(def);
                                    }
                                }
                            } catch(Error t) {
                                LOGGER.error("Failed to adapt {}", interface_.getName(), t);
                                throw t;
                            } catch(RuntimeException t) {
                                LOGGER.error("Failed to adapt {}", interface_.getName(), t);
                                throw t;
                            }
                        }

                    }, fileName);
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    /**
     * Builds cached parameter definitions from a DOM node during cache building.
     */
    private List<CachedParameterDefinition> buildCachedParameters(ReadGraph g, Node node, Loader b) throws DatabaseException, DOMException {
        List<CachedParameterDefinition> cachedParams = new ArrayList<>();
        NodeList nodeList = node.getChildNodes();
        
        for(int i=0;i<nodeList.getLength();++i) {
            Node n = nodeList.item(i);
            if(n.getNodeType() == Node.ELEMENT_NODE) {
                NamedNodeMap attr = n.getAttributes();
                CachedParameterDefinition cached = null;
                
                if(n.getNodeName().equals("this")) {
                    cached = new CachedParameterDefinition(ParameterType.THIS);
                } else if(n.getNodeName().equals("graph")) {
                    cached = new CachedParameterDefinition(ParameterType.GRAPH);
                } else if(n.getNodeName().equals("bundle")) {
                    cached = new CachedParameterDefinition(ParameterType.BUNDLE);
                    Node fc = n.getFirstChild();
                    if (fc != null) {
                        cached.bundleId = fc.getNodeValue();
                    }
                } else if(n.getNodeName().equals("related")) {
                    String uri = attr.getNamedItem("uri").getNodeValue();
                    cached = new CachedParameterDefinition(ParameterType.RELATED, 
                            g.getResource(uri).getResourceId());
                } else if(n.getNodeName().equals("orderedSet")) {
                    String uri = attr.getNamedItem("uri").getNodeValue();
                    cached = new CachedParameterDefinition(ParameterType.ORDERED_SET, 
                            g.getResource(uri).getResourceId());
                } else if(n.getNodeName().equals("single")) {
                    String uri = attr.getNamedItem("uri").getNodeValue();
                    cached = new CachedParameterDefinition(ParameterType.SINGLE, 
                            g.getResource(uri).getResourceId());
                } else if(n.getNodeName().equals("atMostOne")) {
                    String uri = attr.getNamedItem("uri").getNodeValue();
                    cached = new CachedParameterDefinition(ParameterType.AT_MOST_ONE, 
                            g.getResource(uri).getResourceId());
                } else if(n.getNodeName().equals("string")) {
                    cached = new CachedParameterDefinition(ParameterType.STRING, 
                            n.getFirstChild().getNodeValue());
                }
                
                if (cached != null) {
                    Node toNode = attr.getNamedItem("to");
                    if(toNode != null) {
                        cached.adaptTo = toNode.getNodeValue();
                    }
                    cachedParams.add(cached);
                }
            }
        }
        
        return cachedParams;
    }
    
    private List<IDynamicAdapter2> readParameters(ReadGraph g, Node node, Loader b) throws DatabaseException, DOMException, ClassNotFoundException {
        NodeList nodeList = node.getChildNodes();
        ArrayList<IDynamicAdapter2> parameters = new ArrayList<IDynamicAdapter2>();
        for(int i=0;i<nodeList.getLength();++i) {
            Node n = nodeList.item(i);
            if(n.getNodeType() == Node.ELEMENT_NODE) {
                NamedNodeMap attr = n.getAttributes();
                IDynamicAdapter2 da = null;
                if(n.getNodeName().equals("this"))
                    da = ThisResource2.INSTANCE;
                else if(n.getNodeName().equals("graph"))
                    da = GraphObject2.INSTANCE;
                else if(n.getNodeName().equals("bundle")) {
                    String bundleId = null;
                    Node fc = n.getFirstChild();
                    if (fc != null)
                        bundleId = fc.getNodeValue();
                    if (bundleId == null) {
                        da = new ConstantAdapter(Bundle.class, b.getBundle());
                    } else {
                        Bundle ob = Platform.getBundle(bundleId);
                        if (ob != null) {
                            da = new ConstantAdapter(Bundle.class, ob);
                        } else {
                            throw new DOMException(DOMException.NOT_FOUND_ERR, "bundle '" + bundleId + "' not found");
                        }
                    }
                } else if(n.getNodeName().equals("related"))
                    da = new RelatedResources2(
                            g.getResource(attr.getNamedItem("uri").getNodeValue()));
                else if(n.getNodeName().equals("orderedSet"))
                    da = new OrderedSetResources2(
                            g.getResource(attr.getNamedItem("uri").getNodeValue()));
                else if(n.getNodeName().equals("single"))
                    da = new SingleRelatedResource2(
                            g.getResource(attr.getNamedItem("uri").getNodeValue()));
                else if(n.getNodeName().equals("atMostOne"))
                    da = new AtMostOneRelatedResource2(
                            g.getResource(attr.getNamedItem("uri").getNodeValue()));
                else if(n.getNodeName().equals("string"))
                    da = new ConstantAdapter(String.class, n.getFirstChild().getNodeValue());
                {
                    Node toNode = attr.getNamedItem("to");
                    if(toNode != null) {
                        String to = toNode.getNodeValue();
                        da = new AdaptingDynamicAdapter2(da, b.loadClass(to));
                    }
                }
                parameters.add(da);
            }
        }
        return parameters;
    }

    private <T> void handleAdapter(final Loader b, final Class<T> interface_, Node node, String fileName) {
        try {
            NamedNodeMap attr = node.getAttributes();
            final String uri = attr.getNamedItem(URI).getNodeValue();
            final String clazz = attr.getNamedItem(ADAPTER_CLASS).getNodeValue();

            Node contextNode = attr.getNamedItem(CONTEXT_CLASS);
            final String contextClassName = contextNode != null ? contextNode.getNodeValue() : null;
            final Class<?> contextClass = contextClassName != null ? b.loadClass(contextClassName) : Resource.class;
            
            //System.out.println("AdapterRegistry2.handleAdapter: " + b + " " + uri + " " + interface_ + ", class=" + clazz);
            addInstaller(
                    new AdapterInstaller() {

                        @SuppressWarnings("unchecked")
                        @Override
                        public void install(ReadGraph g, AdaptionService service) throws Exception {
                            Resource typeResource = g.getResource(uri);
                            service.addAdapter(
                                    typeResource,
                                    interface_,
                                    contextClass,
                                    ((Class<?>)b.loadClass(clazz))
                                    .asSubclass(Adapter.class).getDeclaredConstructor().newInstance());
                            
                            // Build cache definition if in cache building mode
                            if (buildingCache) {
                                CachedAdapterDefinition def = CachedAdapterDefinition.forAdapterClass(
                                        typeResource.getResourceId(),
                                        interface_.getName(),
                                        clazz,
                                        contextClassName,
                                        b.getBundle() != null ? b.getBundle().getSymbolicName() : "unknown",
                                        fileName);
                                synchronized (cacheDefinitions) {
                                    cacheDefinitions.add(def);
                                }
                            }
                        }

                    }, fileName);
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    private <T> void handleBaseType(Loader b, final Class<T> interface_, Node node, String fileName) {
        try {
            NamedNodeMap attr = node.getAttributes();
            final String uri = attr.getNamedItem(URI).getNodeValue();
            
            // Create a caching installer that captures resource IDs
            AdapterInstaller installer = new AdapterInstaller() {
                @Override
                public void install(ReadGraph g, AdaptionService service) throws Exception {
                    Resource typeResource = g.getResource(uri);
                    service.declareAdapter(typeResource, interface_);
                    
                    // Build cache definition if in cache building mode
                    if (buildingCache) {
                        CachedAdapterDefinition def = CachedAdapterDefinition.forBaseType(
                                typeResource.getResourceId(),
                                interface_.getName(),
                                b.getBundle() != null ? b.getBundle().getSymbolicName() : "unknown",
                                fileName);
                        synchronized (cacheDefinitions) {
                            cacheDefinitions.add(def);
                        }
                    }
                }
            };
            
            addInstaller(installer, fileName);
        } catch (Exception e) {
            handleException(e, fileName);
        }
    }

    public void updateAdaptionService(Session s, final AdaptionService service, AdapterCacheManager cacheManager, boolean initializedFromCache) throws DatabaseException {
        s.syncRequest((ReadGraph g) -> {
            var keys = installerSources.keySet().stream();

            if (initializedFromCache) {
                // NOTE: this parallelization is not directly OK, because it
                // schedules the use of ReadGraph into threads external to the
                // DB transaction thread, but in this case it does not matter
                // because the cached AdapterInstallers only use ReadGraph for
                // SerialisationSupport and resource deserialization.
                keys = keys.parallel();
            }

            keys.forEach(t -> {
                try {
                    t.install(g, service);
                } catch (Throwable e) {
                    AdapterRegistry2.this.handleException(e, t);
                }
            });

            return null;
        });

        // Save cache if cache manager was provided
        if (buildingCache && cacheManager != null && !cacheDefinitions.isEmpty()) {
            buildingCache = false;
            try {
                cacheManager.saveCache(cacheDefinitions);
            } catch (Exception e) {
                LOGGER.error("Failed to save adapter cache", e);
            }
        }
    }

    /**
     * Initializes the adapter registry from a cache.
     * This is much faster than XML parsing as it avoids DOM parsing and URI resolution.
     * 
     * @param context the bundle context
     * @param cacheManager the cache manager containing cached definitions
     * @throws Exception if initialization fails
     */
    public void initializeFromCache(BundleContext context, AdapterCacheManager cacheManager) throws Exception {
        LOGGER.info("Initializing from cache");

        Map<String, Bundle> bundleMap = getBundleMap(context);
        LOGGER.info("Initialized bundle map");
        List<CachedAdapterDefinition> definitions = cacheManager.loadCache();
        LOGGER.info("Loaded cache file");

        for (CachedAdapterDefinition def : definitions) {
            try {
                AdapterInstaller installer = createInstallerFromCache(def, bundleMap);
                if (installer != null) {
                    addInstaller(installer, def.sourceFile);
                }
            } catch (Exception e) {
                LOGGER.error("Failed to create installer from cached definition: {}", def, e);
            }
        }

        LOGGER.info("Initialized {} adapters from cache", installerSources.size());
    }

    /**
     * Creates an AdapterInstaller from a cached definition.
     * @param bundleMap 
     */
    private AdapterInstaller createInstallerFromCache(CachedAdapterDefinition def, Map<String, Bundle> bundleMap) throws Exception {
        Bundle bundle = bundleMap.get(def.bundleSymbolicName);
        if (bundle == null) {
            LOGGER.warn("Bundle not found for cached adapter: {}", def.bundleSymbolicName);
            return null;
        }

        Loader loader = loader(bundle);

        switch (def.type) {
            case BASE_TYPE:
                return createBaseTypeInstallerFromCache(def, loader);
            case TYPE:
                return createTypeInstallerFromCache(def, loader);
            case RESOURCE:
                return createResourceInstallerFromCache(def, loader);
            case ADAPTER_CLASS:
                return createAdapterClassInstallerFromCache(def, loader);
            case INSTALLER:
                return createCustomInstallerFromCache(def, loader);
            default:
                throw new IllegalArgumentException("Unknown adapter type: " + def.type);
        }
    }

    private Map<String, Bundle> getBundleMap(BundleContext context) {
        var result = new HashMap<String, Bundle>();
        for (Bundle bundle : context.getBundles()) {
            result.put(bundle.getSymbolicName(), bundle);
        }
        return result;
    }

    private static Resource getResourceById(ReadGraph g, long id) throws DatabaseException {
        return g.getService(SerialisationSupport.class).getResource(id);
    }

    private AdapterInstaller createBaseTypeInstallerFromCache(CachedAdapterDefinition def, Loader loader) throws Exception {
        Class<?> targetInterface = loader.loadClass(def.targetInterface);
        
        return new AdapterInstaller() {
            @Override
            public void install(ReadGraph g, AdaptionService service) throws Exception {
                Resource typeResource = getResourceById(g, def.typeResourceId);
                service.declareAdapter(typeResource, targetInterface);
            }
        };
    }
    
    @SuppressWarnings("unchecked")
    private <T> AdapterInstaller createTypeInstallerFromCache(CachedAdapterDefinition def, Loader loader) throws Exception {
        Class<T> targetInterface = (Class<T>) loader.loadClass(def.targetInterface);
        
        return new AdapterInstaller() {
            @Override
            public void install(ReadGraph g, AdaptionService service) throws Exception {
                Class<? extends T> adapterClass = loader.loadClass(def.adapterClass).asSubclass(targetInterface);
                IDynamicAdapter2[] parameters = createParametersFromCache(g, def.parameters, loader);
                Resource typeResource = getResourceById(g, def.typeResourceId);
                
                if (def.constructor == null) {
                    service.addAdapter(typeResource, targetInterface, new ReflectionAdapter2<T>(adapterClass, parameters));
                } else {
                    service.addAdapter(typeResource, targetInterface, new StaticMethodAdapter<T>(adapterClass, def.constructor, parameters));
                }
            }
        };
    }
    
    @SuppressWarnings("unchecked")
    private <T> AdapterInstaller createResourceInstallerFromCache(CachedAdapterDefinition def, Loader loader) throws Exception {
        Class<T> targetInterface = (Class<T>) loader.loadClass(def.targetInterface);
        
        return new AdapterInstaller() {
            @Override
            public void install(ReadGraph g, AdaptionService service) throws Exception {
                Class<? extends T> adapterClass = loader.loadClass(def.adapterClass).asSubclass(targetInterface);
                IDynamicAdapter2[] parameters = createParametersFromCache(g, def.parameters, loader);
                Resource resource = getResourceById(g, def.typeResourceId);
                
                if (def.constructor == null) {
                    service.addInstanceAdapter(resource, targetInterface, new ReflectionAdapter2<T>(adapterClass, parameters));
                } else {
                    service.addInstanceAdapter(resource, targetInterface, new StaticMethodAdapter<T>(adapterClass, def.constructor, parameters));
                }
            }
        };
    }
    
    @SuppressWarnings("unchecked")
    private <T> AdapterInstaller createAdapterClassInstallerFromCache(CachedAdapterDefinition def, Loader loader) throws Exception {
        Class<T> targetInterface = (Class<T>) loader.loadClass(def.targetInterface);
        Class<?> contextClass = def.contextClass != null ? loader.loadClass(def.contextClass) : Resource.class;
        
        return new AdapterInstaller() {
            @Override
            public void install(ReadGraph g, AdaptionService service) throws Exception {
                Class<?> adapterClass = loader.loadClass(def.adapterClass);
                Adapter<T, ?> adapter = (Adapter<T, ?>) adapterClass.asSubclass(Adapter.class).getDeclaredConstructor().newInstance();
                Resource typeResource = getResourceById(g, def.typeResourceId);
                service.addAdapter(typeResource, targetInterface, contextClass, adapter);
            }
        };
    }
    
    private AdapterInstaller createCustomInstallerFromCache(CachedAdapterDefinition def, Loader loader) throws Exception {
        return (AdapterInstaller) loader.loadClass(def.installerClassName).asSubclass(AdapterInstaller.class).getDeclaredConstructor().newInstance();
    }
    
    private IDynamicAdapter2[] createParametersFromCache(ReadGraph g, List<CachedParameterDefinition> paramDefs, Loader loader) throws Exception {
        if (paramDefs == null || paramDefs.isEmpty()) {
            return new IDynamicAdapter2[0];
        }
        
        IDynamicAdapter2[] parameters = new IDynamicAdapter2[paramDefs.size()];
        SerialisationSupport ss = g.getService(SerialisationSupport.class);
        
        for (int i = 0; i < paramDefs.size(); i++) {
            CachedParameterDefinition paramDef = paramDefs.get(i);
            IDynamicAdapter2 param = null;
            
            switch (paramDef.type) {
                case THIS:
                    param = ThisResource2.INSTANCE;
                    break;
                case GRAPH:
                    param = GraphObject2.INSTANCE;
                    break;
                case BUNDLE:
                    if (paramDef.bundleId == null) {
                        param = new ConstantAdapter(Bundle.class, loader.getBundle());
                    } else {
                        Bundle bundle = Platform.getBundle(paramDef.bundleId);
                        if (bundle != null) {
                            param = new ConstantAdapter(Bundle.class, bundle);
                        } else {
                            throw new Exception("Bundle not found: " + paramDef.bundleId);
                        }
                    }
                    break;
                case RELATED:
                    param = new RelatedResources2(ss.getResource(paramDef.relationResourceId));
                    break;
                case ORDERED_SET:
                    param = new OrderedSetResources2(ss.getResource(paramDef.relationResourceId));
                    break;
                case SINGLE:
                    param = new SingleRelatedResource2(ss.getResource(paramDef.relationResourceId));
                    break;
                case AT_MOST_ONE:
                    param = new AtMostOneRelatedResource2(ss.getResource(paramDef.relationResourceId));
                    break;
                case STRING:
                    param = new ConstantAdapter(String.class, paramDef.constantValue);
                    break;
            }
            
            if (param != null && paramDef.adaptTo != null) {
                param = new AdaptingDynamicAdapter2(param, loader.loadClass(paramDef.adaptTo));
            }
            
            parameters[i] = param;
        }
        
        return parameters;
    }

    public void initialize(ClassLoader b, String schemaURL, File[] files) {

        try {
        	
            DocumentBuilderFactory factory =
                DocumentBuilderFactory.newInstance();
            
            if(schemaURL != null && validateAgainstSchema()) {
            
	            factory.setValidating(true);
	            factory.setAttribute(
	                    "http://java.sun.com/xml/jaxp/properties/schemaLanguage",
	            "http://www.w3.org/2001/XMLSchema");
	            factory.setAttribute(
	                    "http://java.sun.com/xml/jaxp/properties/schemaSource", schemaURL);
	            
            }

            // TODO Listen bundles (install/uninstall)
            if (exceptions.isEmpty())
                for (final File f : files) {
//                        String fileName = new Path(b.getLocation()).append(file.getPath()).toString();
                        try {
                            DocumentBuilder builder = factory.newDocumentBuilder();
                            builder.setErrorHandler(new ErrorHandler() {

                                @Override
                                public void error(SAXParseException exception)
                                throws SAXException {
                                    // TODO Put this error somewhere
                                    System.err.println("Parse error at " + f.getAbsolutePath() + 
//                                            + b.getSymbolicName() + "/adapters.xml" +
                                            " line " + exception.getLineNumber() +
                                            " column " + exception.getColumnNumber() + ":");
                                    System.err.println(exception.getMessage());
                                }

                                @Override
                                public void fatalError(SAXParseException exception)
                                throws SAXException {
                                    error(exception);
                                }

                                @Override
                                public void warning(SAXParseException exception)
                                throws SAXException {
                                    error(exception);
                                }

                            });
                            //System.out.println("bundle=" + b.getSymbolicName());
                            Document doc = builder.parse(f);
                            handleAdaptersDocument(loader(b), doc, f.getAbsolutePath());
                        } catch (Exception e) {
                            handleException(e, f.getAbsolutePath());

                        }
                    }
        } catch (Exception e) {
            handleException(e, "(no file name available)");
        }
    	
    }

    private boolean validateAgainstSchema() {
        return Platform.inDevelopmentMode();
    }

    /**
     * Initializes the adapter registry from adapters.xml files and builds the cache.
     * 
     * @param context the bundle context
     * @param cacheManager optional cache manager to save the cache after initialization, or null to skip caching
     */
    public void initialize(BundleContext context, AdapterCacheManager cacheManager) {
        LOGGER.info("Initializing from XML");
        buildingCache = cacheManager != null;
        cacheDefinitions.clear();
        try {
        	
            DocumentBuilderFactory factory =
                DocumentBuilderFactory.newInstance();

            if (validateAgainstSchema()) {
                factory.setValidating(true);
                factory.setAttribute(
                        "http://java.sun.com/xml/jaxp/properties/schemaLanguage",
                        "http://www.w3.org/2001/XMLSchema");
                factory.setAttribute(
                        "http://java.sun.com/xml/jaxp/properties/schemaSource",
                        context.getBundle().getResource("adapters.xsd").toString());
            }

            // TODO Listen bundles (install/uninstall)
            List<Future<?>> waitFor = new ArrayList<>();
            if (exceptions.isEmpty())
                for (final Bundle b : context.getBundles()) {
                    Future<?> submit = ThreadUtils.getNonBlockingWorkExecutor().submit(() -> {
                        URL file = b.getEntry(ADAPTERS_FILE);
                        if (file != null) {
                            String fileName = new Path(b.getLocation()).append(file.getPath()).toString();
                            try {
                                DocumentBuilder builder = factory.newDocumentBuilder();
                                builder.setErrorHandler(new ErrorHandler() {

                                    @Override
                                    public void error(SAXParseException exception) throws SAXException {
                                        // TODO Put this error somewhere
                                        System.err.println("Parse error at " + b.getSymbolicName() + "/adapters.xml"
                                                + " line " + exception.getLineNumber() + " column "
                                                + exception.getColumnNumber() + ":");
                                        System.err.println(exception.getMessage());
                                    }

                                    @Override
                                    public void fatalError(SAXParseException exception) throws SAXException {
                                        error(exception);
                                    }

                                    @Override
                                    public void warning(SAXParseException exception) throws SAXException {
                                        error(exception);
                                    }

                                });

                                // System.out.println("bundle=" + b.getSymbolicName());
                                String text = FileUtils.getContents(file);
                                text = OntologyVersions.getInstance().currentVersion(text);
                                StringReader reader = new StringReader(text);
                                InputSource inputSource = new InputSource(reader);
                                Document doc = builder.parse(inputSource);
                                reader.close();
                                handleAdaptersDocument(loader(b), doc, fileName);
                            } catch (Exception e) {
                                handleException(e, fileName);

                            }
                        }
                    });
                    waitFor.add(submit);
                }
            // Let's wait in here
            waitFor.forEach(f -> {
                try {
                    f.get();
                } catch (InterruptedException | ExecutionException e) {
                    LOGGER.error("Could not wait adapters to load", e);
                }
            });
            LOGGER.info("Adapters installed");
        } catch (Exception e) {
            handleException(e, "(no file name available)");
        }
    }

    /**
     * Initializes the adapter registry from adapters.xml files without caching.
     * This is the original method signature kept for backward compatibility.
     * 
     * @param context the bundle context
     */
    public void initialize(BundleContext context) {
        initialize(context, null);
    }

	interface Loader {
		Class<?> loadClass(String name) throws ClassNotFoundException ;
		Bundle getBundle();
	}

	private Loader loader(final Bundle b) {
		return new Loader() {

			@Override
			public Class<?> loadClass(String name) throws ClassNotFoundException {
				return b.loadClass(name);
			}
			
			@Override
			public Bundle getBundle() {
				return b;
			}
			
		};
	}

	private Loader loader(final ClassLoader b) {
		return new Loader() {

			@Override
			public Class<?> loadClass(String name) throws ClassNotFoundException {
				return b.loadClass(name);
			}

			@Override
			public Bundle getBundle() {
				return null;
			}
			
		};
	}
	
}
