/*******************************************************************************
 * Copyright (c) 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.services.adaption;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Externalizable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.simantics.db.services.internal.Activator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Manages the adapter registry cache including validation, loading, and saving.
 * 
 * <p>The cache stores pre-resolved adapter definitions with resource IDs to avoid
 * expensive XML parsing and URI resolution during startup.</p>
 * 
 * <p>Cache invalidation happens when:</p>
 * <ul>
 *   <li>Any adapters.xml file has been modified</li>
 *   <li>Ontology installation/merge has occurred</li>
 *   <li>Cache file is corrupted or has incompatible version</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * @since 1.67.0
 */
public class AdapterCacheManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdapterCacheManager.class);

    private static final String CACHE_FILE_NAME = "adapter-cache.bin";
    private static final String TIMESTAMPS_FILE_NAME = "adapter-timestamps.properties";
    private static final String ONTOLOGY_MARKER_FILE_NAME = "ontology-version.txt";
    private static final int CACHE_FORMAT_VERSION = 2;

    private final BundleContext context;
    private final Path cacheDir;

    /**
     * Wrapper class for serializing cache data with version information.
     * Uses custom serialization with string table for maximum performance and space efficiency.
     */
    private static class CacheData implements Externalizable {

        private static final long serialVersionUID = 2L;

        private int formatVersion;
        private long timestamp;
        private List<CachedAdapterDefinition> definitions;

        public CacheData() {}

        public CacheData(int formatVersion, long timestamp, List<CachedAdapterDefinition> definitions) {
            this.formatVersion = formatVersion;
            this.timestamp = timestamp;
            this.definitions = definitions;
        }

        /**
         * Custom serialization with string table for deduplication.
         * This provides maximum performance by eliminating duplicate strings.
         */
        @Override
        public void writeExternal(ObjectOutput out) throws IOException {
            // Write format version and timestamp
            out.writeInt(formatVersion);
            out.writeLong(timestamp);

            // Build string table from all definitions
            StringTable stringTable = new StringTable();
            for (CachedAdapterDefinition def : definitions) {
                stringTable.add(def.targetInterface);
                stringTable.add(def.adapterClass);
                stringTable.add(def.contextClass);
                stringTable.add(def.constructor);
                stringTable.add(def.installerClassName);
                stringTable.add(def.bundleSymbolicName);
                stringTable.add(def.sourceFile);

                if (def.parameters != null) {
                    for (CachedParameterDefinition param : def.parameters) {
                        stringTable.add(param.constantValue);
                        stringTable.add(param.bundleId);
                        stringTable.add(param.adaptTo);
                    }
                }
            }

            // Write string table
            stringTable.write(out);

            // Write definitions count
            out.writeInt(definitions.size());

            // Write each definition using string table
            for (CachedAdapterDefinition def : definitions) {
                writeDefinitionWithStringTable(out, def, stringTable);
            }
        }

        /**
         * Writes a definition using string table indices instead of full strings
         */
        private void writeDefinitionWithStringTable(ObjectOutput out, 
                CachedAdapterDefinition def, StringTable stringTable) throws IOException {

            // Write enum ordinal
            out.writeByte(def.type.ordinal());

            // Write type resource ID
            out.writeLong(def.typeResourceId);

            // Write string indices (using -1 for null)
            out.writeInt(stringTable.getIndex(def.targetInterface));
            out.writeInt(stringTable.getIndex(def.adapterClass));
            out.writeInt(stringTable.getIndex(def.contextClass));
            out.writeInt(stringTable.getIndex(def.constructor));
            out.writeInt(stringTable.getIndex(def.installerClassName));
            out.writeInt(stringTable.getIndex(def.bundleSymbolicName));
            out.writeInt(stringTable.getIndex(def.sourceFile));

            // Write parameters
            int paramCount = (def.parameters != null) ? def.parameters.size() : 0;
            out.writeInt(paramCount);
            for (int i = 0; i < paramCount; i++) {
                writeParameterWithStringTable(out, def.parameters.get(i), stringTable);
            }
        }

        /**
         * Writes a parameter using string table indices
         */
        private void writeParameterWithStringTable(ObjectOutput out,
                CachedParameterDefinition param, StringTable stringTable) throws IOException {

            // Write enum ordinal
            out.writeByte(param.type.ordinal());

            // Write relation resource ID
            out.writeLong(param.relationResourceId);

            // Write string indices
            out.writeInt(stringTable.getIndex(param.constantValue));
            out.writeInt(stringTable.getIndex(param.bundleId));
            out.writeInt(stringTable.getIndex(param.adaptTo));
        }

        /**
         * Custom deserialization with string table
         */
        @Override
        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
            formatVersion = in.readInt();
            timestamp = in.readLong();

            // Read string table
            var stringTable = new StringTable();
            stringTable.read(in);

            // Read definitions count
            int defCount = in.readInt();
            var defs = new ArrayList<CachedAdapterDefinition>(defCount);

            // Read each definition
            for (int i = 0; i < defCount; i++) {
                defs.add(readDefinitionWithStringTable(in, stringTable));
            }

            definitions = defs;
        }

        /**
         * Reads a definition using string table
         */
        private CachedAdapterDefinition readDefinitionWithStringTable(ObjectInput in, StringTable stringTable) throws IOException {

            var def = new CachedAdapterDefinition();

            // Read enum ordinal
            def.type = AdapterType.values()[in.readByte()];

            // Read type resource ID
            def.typeResourceId = in.readLong();

            // Read strings from table
            def.targetInterface = stringTable.getString(in.readInt());
            def.adapterClass = stringTable.getString(in.readInt());
            def.contextClass = stringTable.getString(in.readInt());
            def.constructor = stringTable.getString(in.readInt());
            def.installerClassName = stringTable.getString(in.readInt());
            def.bundleSymbolicName = stringTable.getString(in.readInt());
            def.sourceFile = stringTable.getString(in.readInt());

            // Read parameters
            int paramCount = in.readInt();
            def.parameters = new ArrayList<>(paramCount);
            for (int i = 0; i < paramCount; i++) {
                def.parameters.add(readParameterWithStringTable(in, stringTable));
            }

            return def;
        }

        /**
         * Reads a parameter using string table
         */
        private CachedParameterDefinition readParameterWithStringTable(ObjectInput in, StringTable stringTable) throws IOException {

            var param = new CachedParameterDefinition();

            // Read enum ordinal
            param.type = ParameterType.values()[in.readByte()];

            // Read relation resource ID
            param.relationResourceId = in.readLong();

            // Read strings from table
            param.constantValue = stringTable.getString(in.readInt());
            param.bundleId = stringTable.getString(in.readInt());
            param.adaptTo = stringTable.getString(in.readInt());

            return param;
        }

    }

    /**
     * String table for deduplicating repeated strings during serialization.
     * Provides significant space savings when many definitions share common strings.
     */
    private static class StringTable {
        private final Map<String, Integer> stringToIndex = new HashMap<>();
        private final List<String> indexToString = new ArrayList<>();

        /**
         * Adds a string to the table if not null and not already present
         */
        void add(String str) {
            if (str != null && !stringToIndex.containsKey(str)) {
                int index = indexToString.size();
                stringToIndex.put(str, index);
                indexToString.add(str);
            }
        }

        /**
         * Gets the index of a string, or -1 if null
         */
        int getIndex(String str) {
            if (str == null) {
                return -1;
            }
            Integer index = stringToIndex.get(str);
            return (index != null) ? index : -1;
        }

        /**
         * Gets a string by index, or null if index is -1
         */
        String getString(int index) {
            return (index >= 0) ? indexToString.get(index) : null;
        }

        /**
         * Writes the string table to the output stream
         */
        void write(ObjectOutput out) throws IOException {
            out.writeInt(indexToString.size());
            for (String str : indexToString) {
                out.writeUTF(str);
            }
        }

        /**
         * Reads the string table from the input stream
         */
        void read(ObjectInput in) throws IOException {
            int size = in.readInt();
            indexToString.clear();
            stringToIndex.clear();

            for (int i = 0; i < size; i++) {
                String str = in.readUTF();
                indexToString.add(str);
                stringToIndex.put(str, i);
            }
        }
    }

    /**
     * Creates a cache manager for the given bundle context.
     * 
     * @param context the OSGi bundle context
     */
    public AdapterCacheManager(BundleContext context) {
        this.context = context;
        this.cacheDir = getCacheDirectory();

        try {
            Files.createDirectories(cacheDir);
        } catch (IOException e) {
            LOGGER.error("Failed to create cache directory", e);
        }
    }

    /**
     * Gets the platform state location directory for cache files.
     */
    private static Path getCacheDirectory() {
        try {
            Path stateLocation = Platform.getStateLocation(Activator.getDefault().getBundle()).toFile().toPath();
            return stateLocation.resolve("adapter-cache");
        } catch (Exception e) {
            LOGGER.warn("Could not get platform state location, using temp directory", e);
            return Paths.get(System.getProperty("java.io.tmpdir"), "simantics-adapter-cache");
        }
    }

    private Path getCacheFile() {
        return cacheDir.resolve(CACHE_FILE_NAME);
    }

    private Path getTimestampsFile() {
        return cacheDir.resolve(TIMESTAMPS_FILE_NAME);
    }

    private Path getOntologyMarkerFile() {
        return cacheDir.resolve(ONTOLOGY_MARKER_FILE_NAME);
    }

    /**
     * Checks if the cache is valid and can be used for initialization.
     * 
     * @return true if cache is valid, false otherwise
     */
    public boolean isCacheValid() {
        // Check if caching is disabled via system property
        if ("false".equals(System.getProperty("simantics.adapters.cache.enabled"))) {
            LOGGER.info("Adapter cache disabled via system property");
            return false;
        }

        Path cacheFile = getCacheFile();
        Path timestampsFile = getTimestampsFile();

        // Check files exist
        if (!Files.exists(cacheFile)) {
            LOGGER.debug("Cache file does not exist");
            return false;
        }

        if (!Files.exists(timestampsFile)) {
            LOGGER.debug("Timestamps file does not exist");
            return false;
        }

        // Check ontology changes (conservative strategy)
        try {
            Path ontologyMarkerFile = getOntologyMarkerFile();
            if (Files.exists(ontologyMarkerFile) && 
                Files.getLastModifiedTime(ontologyMarkerFile).toMillis() > 
                Files.getLastModifiedTime(cacheFile).toMillis()) {
                LOGGER.info("Cache invalid: ontology changes detected");
                return false;
            }
        } catch (IOException e) {
            LOGGER.warn("Failed to check ontology marker file", e);
            return false;
        }

        try {
            // Check adapters.xml files
            Map<String, FileChecksum> currentChecksums = computeAdapterFileChecksums();
            Map<String, FileChecksum> cachedChecksums = loadTimestamps();

            // Quick check: if bundle count differs, cache is invalid
            if (currentChecksums.size() != cachedChecksums.size()) {
                LOGGER.info("Cache invalid: number of bundles with adapters.xml changed ({} -> {})",
                        cachedChecksums.size(), currentChecksums.size());
                return false;
            }

            // Check each adapter file
            for (Map.Entry<String, FileChecksum> entry : currentChecksums.entrySet()) {
                String bundleId = entry.getKey();
                FileChecksum current = entry.getValue();
                FileChecksum cached = cachedChecksums.get(bundleId);

                if (cached == null) {
                    LOGGER.info("Cache invalid: new bundle with adapters.xml: {}", bundleId);
                    return false;
                }

                if (current.lastModified > cached.lastModified) {
                    LOGGER.info("Cache invalid: adapters.xml modified in bundle: {}", bundleId);
                    return false;
                }

                if (!current.md5.equals(cached.md5)) {
                    LOGGER.info("Cache invalid: adapters.xml checksum changed in bundle: {}", bundleId);
                    return false;
                }
            }

            LOGGER.info("Cache is valid");
            return true;
        } catch (Exception e) {
            LOGGER.warn("Error validating cache", e);
            return false;
        }
    }

    /**
     * Loads cached adapter definitions from disk.
     * 
     * @return list of cached adapter definitions
     * @throws CacheInvalidException if cache cannot be loaded
     */
    public List<CachedAdapterDefinition> loadCache() throws CacheInvalidException {
        Path cacheFile = getCacheFile();

        if (!Files.exists(cacheFile)) {
            throw new CacheInvalidException("Cache file does not exist");
        }

        try (InputStream is = Files.newInputStream(cacheFile);
             BufferedInputStream bis = new BufferedInputStream(is, (1 << 17));
             ObjectInputStream ois = new ObjectInputStream(bis)) {

            CacheData data = new CacheData();
            data.readExternal(ois);

            if (data.formatVersion != CACHE_FORMAT_VERSION) {
                throw new CacheInvalidException("Cache format version mismatch: expected " + 
                        CACHE_FORMAT_VERSION + ", got " + data.formatVersion);
            }

            LOGGER.info("Loaded {} adapter definitions from cache", data.definitions.size());
            return data.definitions;

        } catch (IOException | ClassNotFoundException e) {
            throw new CacheInvalidException("Failed to load cache file", e);
        }
    }

    /**
     * Saves adapter definitions to the cache.
     * 
     * @param definitions the adapter definitions to cache
     */
    public void saveCache(List<CachedAdapterDefinition> definitions) {
        Path cacheFile = getCacheFile();

        try {
            // Ensure cache directory exists
            Files.createDirectories(cacheDir);

            // Save cache data
            CacheData data = new CacheData(CACHE_FORMAT_VERSION, System.currentTimeMillis(), definitions);

            try (OutputStream os = Files.newOutputStream(cacheFile);
                 BufferedOutputStream bos = new BufferedOutputStream(os, (1 << 17));
                 ObjectOutputStream oos = new ObjectOutputStream(bos)) {
                data.writeExternal(oos);
            }

            // Save timestamps & expected hashes
            Map<String, FileChecksum> checksums = computeAdapterFileChecksums();
            saveTimestamps(checksums);

            LOGGER.info("Saved {} adapter definitions to cache", definitions.size());

        } catch (IOException e) {
            LOGGER.error("Failed to save cache", e);
            // Don't throw - cache saving is optional
        }
    }

    /**
     * Invalidates the cache by deleting cache files.
     */
    public void invalidateCache() {
        Path cacheFile = getCacheFile();
        Path timestampsFile = getTimestampsFile();

        try {
            if (Files.deleteIfExists(cacheFile)) {
                LOGGER.info("Deleted cache file");
            }
            if (Files.deleteIfExists(timestampsFile)) {
                LOGGER.info("Deleted timestamps file");
            }
        } catch (IOException e) {
            LOGGER.warn("Failed to delete cache files", e);
        }
    }

    /**
     * Marks that ontology changes have occurred, invalidating the cache.
     * This should be called after successful ontology installation/merge.
     */
    public static void markOntologyChange() {
        try {
            Path cacheDir = getCacheDirectory();

            Files.createDirectories(cacheDir);

            Path markerFile = cacheDir.resolve(ONTOLOGY_MARKER_FILE_NAME);

            String marker = String.valueOf(System.currentTimeMillis());
            Files.write(markerFile, marker.getBytes());

            LOGGER.info("Marked ontology change, adapter cache will be invalidated on next startup");
        } catch (Exception e) {
            LOGGER.warn("Failed to mark ontology change", e);
        }
    }

    private static record BundleURL(Bundle bundle, URL file, Path path, long lastModified) {}

    /**
     * Stores file checksum data.
     */
    private static record FileChecksum(long lastModified, String md5) {}

    /**
     * Computes MD5 checksums for all adapters.xml files in the current environment.
     */
    private Map<String, FileChecksum> computeAdapterFileChecksums() {
        Map<String, FileChecksum> checksums = new ConcurrentHashMap<>();

        // Load previous checksums to enable optimization
        Map<String, FileChecksum> prevChecksums = loadTimestamps();

        BiConsumer<Bundle, Consumer<BundleURL>> adapterResolver = (b, c) -> {
            URL entry = b.getEntry(AdapterRegistry2.ADAPTERS_FILE);
            if (entry != null) {
                try {
                    URL fileUrl = FileLocator.toFileURL(entry);
                    if (fileUrl.getProtocol().equals("file")) { //$NON-NLS-1$
                        File f = new File(URLDecoder.decode(fileUrl.getPath(), "UTF-8")).getAbsoluteFile(); //$NON-NLS-1$
                        Path p = f.toPath();
                        var lastModified = Files.getLastModifiedTime(p);
                        c.accept(new BundleURL(b, fileUrl, p, lastModified.toMillis()));
                    }
                } catch (IOException e) {
                    LOGGER.warn("Cannot read last modification time for {}", entry, e);
                }
            }
        };

        Consumer<BundleURL> checksummer = b -> {
            String bundleId = b.bundle.getSymbolicName();
            try {
                long currentModTime = b.lastModified;
                FileChecksum prev = prevChecksums.get(bundleId);

                // Optimization: reuse cached checksum if modification time hasn't changed
                if (prev != null && prev.lastModified == currentModTime) {
                    checksums.put(bundleId, prev);
                } else {
                    // Compute new checksum
                    String md5 = computeMD5(b.path);
                    checksums.put(bundleId, new FileChecksum(currentModTime, md5));
                }
            } catch (Exception e) {
                LOGGER.warn("Failed to compute checksum for " + bundleId, e);
            }
        };

        Arrays.stream(context.getBundles())
            .parallel()
            .mapMulti(adapterResolver)
            .forEach(checksummer);

        return checksums;
    }

    /**
     * Computes MD5 checksum of a file.
     */
    private String computeMD5(Path p) throws IOException {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] bytes = Files.readAllBytes(p);
            byte[] digest = md.digest(bytes);

            StringBuilder sb = new StringBuilder();
            for (byte b : digest) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException e) {
            throw new Error("MD5 should be supported", e);
        }
    }

    /**
     * Loads timestamps from the properties file.
     */
    private Map<String, FileChecksum> loadTimestamps() {
        Map<String, FileChecksum> checksums = new HashMap<>();
        Path timestampsFile = getTimestampsFile();

        if (!Files.exists(timestampsFile)) {
            return checksums;
        }

        try (InputStream is = Files.newInputStream(timestampsFile)) {
            Properties props = new Properties();
            props.load(is);

            for (String key : props.stringPropertyNames()) {
                if (key.endsWith(".modtime")) {
                    String bundleId = key.substring(0, key.length() - ".modtime".length());
                    long modTime = Long.parseLong(props.getProperty(key));
                    String md5 = props.getProperty(bundleId + ".md5");

                    if (md5 != null) {
                        checksums.put(bundleId, new FileChecksum(modTime, md5));
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.warn("Failed to load timestamps", e);
        }

        return checksums;
    }

    /**
     * Saves timestamps to the properties file.
     */
    private void saveTimestamps(Map<String, FileChecksum> checksums) throws IOException {
        Properties props = new Properties();

        for (Map.Entry<String, FileChecksum> entry : checksums.entrySet()) {
            String bundleId = entry.getKey();
            FileChecksum checksum = entry.getValue();
            props.setProperty(bundleId + ".modtime", String.valueOf(checksum.lastModified));
            props.setProperty(bundleId + ".md5", checksum.md5);
        }

        Path timestampsFile = getTimestampsFile();
        try (OutputStream os = Files.newOutputStream(timestampsFile)) {
            props.store(os, "Adapter file checksums");
        }
    }

}
