/*******************************************************************************
 * Copyright (c) 2007, 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - adapter cache optimization
 *******************************************************************************/
package org.simantics.db.services;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.BundleContext;
import org.simantics.db.Session;
import org.simantics.db.adaption.AdaptionService;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.services.adaption.AdapterCacheManager;
import org.simantics.db.services.adaption.AdapterRegistry2;
import org.simantics.db.services.adaption.AdaptionService2;
import org.simantics.db.services.internal.Activator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 */
public class AdaptionServiceInitializer implements ServiceInitializer {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdaptionServiceInitializer.class);

    public IStatus initialize(final Session s) {
        LOGGER.info("Initializing AdaptionService");
        long startTime = System.currentTimeMillis();

        AdaptionService2 service = new AdaptionService2();
        s.registerService(AdaptionService.class, service);

        try {
            AdapterRegistry2 registry = new AdapterRegistry2();
            BundleContext context = Activator.getDefault().getContext();
            AdapterCacheManager cacheManager = new AdapterCacheManager(context);

            boolean usedCache = false;
            if (cacheManager.isCacheValid()) {
                try {
                    LOGGER.info("Loading adapters from cache");
                    registry.initializeFromCache(context, cacheManager);
                    long elapsed = System.currentTimeMillis() - startTime;
                    LOGGER.info("Adapters loaded from cache in {} ms", elapsed);
                    usedCache = true;
                } catch (Exception e) {
                    LOGGER.warn("Failed to load from cache, falling back to XML parsing", e);
                    usedCache = false;
                }
            }

            if (!usedCache) {
                LOGGER.info("Cache invalid or missing, rebuilding from XML");
                registry = new AdapterRegistry2();
                registry.initialize(context, cacheManager);
                long elapsed = System.currentTimeMillis() - startTime;
                LOGGER.info("Adapters initialized from XML in {} ms", elapsed);
            }

            LOGGER.info("Updating AdaptionService");
            long updateStart = System.currentTimeMillis();
            registry.updateAdaptionService(s, service, cacheManager, usedCache);
            long updateElapsed = System.currentTimeMillis() - updateStart;
            LOGGER.info("AdaptionService updated in {} ms", updateElapsed);

            long totalElapsed = System.currentTimeMillis() - startTime;
            LOGGER.info("Total AdaptionService initialization time: {} ms", totalElapsed);

            return Status.OK_STATUS;
        } catch (DatabaseException e) {
            IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, 
                "AdapterRegistry2 instantiation failed, see exception for details.", e);
            return status;
        }
    }

}