/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.variable;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.BooleanBinding;
import org.simantics.databoard.binding.NumberBinding;
import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.MutableStringBinding;
import org.simantics.databoard.parser.repository.DataTypeSyntaxError;
import org.simantics.databoard.parser.repository.DataValueRepository;
import org.simantics.databoard.primitives.MutableString;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.units.IUnitConverter;
import org.simantics.databoard.units.internal.UnitParseException;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.util.PrimitiveValueParser;

public class VariableWrite extends WriteRequest {

    final private Variable variable;
    final private String label;
    final private String targetUnit;

    public VariableWrite(Variable variable, String label) {
        this(variable, label, null, null);
    }

    public VariableWrite(Variable variable, String label, VirtualGraph provider, String targetUnit) {
        super(provider);
        this.variable = variable;
        this.label = label;
        this.targetUnit = targetUnit;
    }

    public static IUnitConverter converter(final String unit, final String targetUnit) {

        return new IUnitConverter() {

            @Override
            public double convert(double value) {

            	if(targetUnit == null) return value;

                if("K".equals(unit)) {
                    if("C".equals(targetUnit)) value = value - 273.15;
                    else if("F".equals(targetUnit)) value = (value - 273.15) * 9.0/5.0 + 32.0;
                } else if ("C".equals(unit)) {
                    if("K".equals(targetUnit)) value = value + 273.15;
                } else if ("F".equals(unit)) {
                    if("K".equals(targetUnit)) value = (value - 32.0) * 5.0/9.0 + 273.15;
                } else {
                    try {
                        value = org.simantics.databoard.Units.convert(value, unit, targetUnit);
                    } catch (UnitParseException e) {
                        e.printStackTrace();
                    }
                }

                return value;

            }

        };


    }

    @Override
    public void perform(WriteGraph graph) throws DatabaseException {
        graph.markUndoPoint();
        try {
            //System.err.println("VariableWrite " + variable.getURI(graph) + " => " + label);
            Datatype type = variable.getPossibleDatatype(graph);
            if (type == null) {
                String uri = null;
                try {
                    uri = variable.getURI(graph);
                } catch (DatabaseException e) {
                }
                throw new org.simantics.db.exception.BindingException("no datatype for variable " + variable + " (URI=" + uri + ")", null);
            }

            Binding binding = Bindings.getBinding(type);

            String parsedLabel = label;
            Object value = null;

            if (binding instanceof StringBinding) {
                if (binding instanceof MutableStringBinding)
                    value = new MutableString(label);
                else
                    value = label;
            } else if (binding instanceof BooleanBinding) {
                // This parses a bit more than just "true"/"false".
                try {
                    value = PrimitiveValueParser.parseBoolean(label);
                } catch (IllegalArgumentException e) {
                    throw new DatabaseException(e);
                }
            } else {
                if (binding instanceof NumberBinding) {
                    parsedLabel = label.replace(",", ".");
                }

                value = binding.parseValue(parsedLabel, new DataValueRepository());
                
                // FIXME: this is Balas-specific
                if (targetUnit != null) {
                    if (type instanceof NumberType) {
                        String unit = Variables.getPossibleUnit(graph, variable);
                        if (unit != null) {
                            IUnitConverter converter = converter(targetUnit, unit);
                            if(type instanceof FloatType) {
                                float converted = (float)converter.convert(((Number)value).floatValue());
                            	value = binding.parseValue(Float.toString(converted), new DataValueRepository());
                            } else if(type instanceof DoubleType) {
                                double converted = converter.convert(((Number)value).doubleValue());
                            	value = binding.parseValue(Double.toString(converted), new DataValueRepository());
                            } else if(type instanceof IntegerType) {
                            	value = binding.parseValue(Integer.toString(((Number)value).intValue()), new DataValueRepository());
                            }
                        }
                    }
                }
            }
            variable.setValue(graph, value, binding);

            String[] varParts = variable.getParent(graph).getURI(graph).split("/");
            String varName = varParts[varParts.length - 1];
            
            // Add a comment to metadata.
            Layer0Utils.addCommentMetadata(graph, "Modified variable " + varName + " to " + ObjectUtils.toString(value));
            
        } catch (DataTypeSyntaxError e) {
            throw new DatabaseException(e);
        } catch (BindingException e) {
            throw new DatabaseException(e);
        }

    }

}
