/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.variable;

import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.Resource;
import org.simantics.db.layer0.variable.RVI.RVIPart;
import org.simantics.db.layer0.variable.Variables.Role;

/**
 * Utility for building RVI parts.
 * 
 * @author toni.kalajainen
 */
public class RVIBuilder {

	Binding rviBinding;
    List<RVIPart> parts;

    public RVIBuilder(Binding rviBinding)
    {
        parts = new ArrayList<RVIPart>();
    }

    public RVIBuilder(RVI rvi)
    {
        set(rvi);
    }

    public RVIBuilder set( RVI rvi ) 
    {
    	this.rviBinding = rvi.getBinding();
        parts = new ArrayList<RVIPart>( rvi.parts.length + 10 );
        for ( RVIPart part : rvi.parts ) parts.add( part );
        return this;
    }

    public RVIBuilder append( RVI rvi ) 
    {
        for ( RVIPart part : rvi.parts ) parts.add( part );
        return this;
    }

    public RVIBuilder append( RVIPart part )
    {
        parts.add( part );
        return this;
    }

    public RVIBuilder append( Role role, String string )
    {
        parts.add( new RVI.StringRVIPart(role, string) );
        return this;
    }

    public RVIBuilder append( Role role, Resource resource )
    {
        parts.add( new RVI.ResourceRVIPart(role, resource) );
        return this;
    }
    
    public RVIBuilder append( Role role, Resource resource, long mostSignificant, long leastSignificant)
    {
        parts.add( new RVI.GuidRVIPart(role, resource, mostSignificant, leastSignificant) );
        return this;
    }

    /**
     * @param rvi the RVI to truncate
     * @param n number of parts to remove from the end of the RVI
     * @return truncated RVI
     */
    public RVIBuilder removeLast(int n) {
        if (n > parts.size())
            throw new IllegalArgumentException("Requested to remove " + n + " parts from RVI, only " + parts.size()
                    + " parts exist in '" + parts + "'");
        parts = parts.subList(0, parts.size() - n);
        return this;
    }

    public RVIBuilder removeFromFirstRole(Role role) {
        for (int i = 0; i < parts.size(); ++i) {
            if (parts.get(i).getRole() == role) {
                parts = parts.subList(0, i);
                return this;
            }
        }
        return this;
    }

    public RVI toRVI()
    {
        if (parts.isEmpty())
            return RVI.empty( rviBinding );
        RVI rvi = new RVI( rviBinding );
        rvi.parts = (RVIPart[]) parts.toArray( new RVIPart[ parts.size() ] );
        return rvi;
    }

}
