package org.simantics.db.layer0.scl;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.layer0.adapter.PrioritizedFactory;
import org.simantics.db.layer0.internal.SimanticsInternal;
import org.simantics.db.layer0.util.DatabaseExceptionUtils;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.scl.runtime.function.Function1;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCLAction implements ActionFactory, PrioritizedFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(SCLAction.class);

    private final Resource rule;
    private final int priority;

    public SCLAction(ReadGraph graph, Resource rule) throws DatabaseException {
        this.rule = rule;
        Layer0 L0 = graph.l0();
        Integer p = graph.getPossibleRelatedValue(rule, L0.SCLAction_priority);
        this.priority = p != null ? p : 0;
    }

    @Override
    public int priority() {
        return priority;
    }

    static class RuleFunctionRequest extends ResourceRead<Function1<Resource, Object>> {

        protected RuleFunctionRequest(Resource rule) {
            super(rule);
        }

        @Override
        public Function1<Resource, Object> perform(ReadGraph graph) throws DatabaseException {
            Variable ruleVariable = Variables.getVariable(graph, resource);
            Layer0 L0 = Layer0.getInstance(graph);
            return ruleVariable.getPossiblePropertyValue(graph, L0.SCLAction_action);
        }

    }

    static class SCLActionRunnable implements Runnable {

        public final Resource rule;
        public final Resource target;

        public SCLActionRunnable(Resource rule, Resource target) {
            this.rule = rule;
            this.target = target;
        }

        @Override
        public void run() {
            Session s = SimanticsInternal.getSession();
            Resource resource = (Resource) target;
            s.markUndoPoint();
            try {
                Function1<Resource, Object> function = s.syncRequest(new RuleFunctionRequest(rule));
                function.apply(resource);
            } catch (DatabaseException e) {
                LOGGER.error("Error while executing action " + DatabaseExceptionUtils.showResource(s, resource), e);
            }
        }

    }

    @Override
    public Runnable create(final Object target) {
        return new SCLActionRunnable(rule, (Resource) target);
    }

}
