/*******************************************************************************
 * Copyright (c) 2021, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.request.external;

import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.request.ParametrizedPrimitiveRead;
import org.simantics.db.procedure.Listener;
import org.simantics.utils.datastructures.Triple;

/**
 * @author Tuukka Lehtonen
 * @since 1.52.0
 */
public class EclipsePreferenceBooleanPrimitiveRead extends ParametrizedPrimitiveRead<Triple<String,String,Boolean>,Boolean> implements IPreferenceChangeListener {

    private Listener<Boolean> listener;

    public EclipsePreferenceBooleanPrimitiveRead(String plugin, String key, boolean def) {
        super(Triple.make(plugin, key, def));
    }

    @Override
    public void register(ReadGraph graph, Listener<Boolean> procedure) {
        IEclipsePreferences preferenceNode = resolvePreferenceNode();
        if (!procedure.isDisposed() && preferenceNode != null) {
            preferenceNode.addPreferenceChangeListener(this);
        }
        listener = procedure;
        // Use get here since getBoolean can't return null when value does not exist
        String prefValue = preferenceNode.get(parameter.second, null);
        Boolean value = prefValue != null ? "true".equalsIgnoreCase(prefValue) : null; //$NON-NLS-1$
        if (value == null) {
            preferenceNode = resolveDefaultPreferenceNode();
            value = preferenceNode != null ? preferenceNode.getBoolean(parameter.second, parameter.third) : parameter.third;
        }
        procedure.execute(value);
    }

    @Override
    public void preferenceChange(PreferenceChangeEvent event) {
        if (parameter.second.equals(event.getKey())) {
            IEclipsePreferences preferenceNode = resolvePreferenceNode();
            // Use get here since getBoolean can't return null when value does not exist
            String prefValue = preferenceNode.get(parameter.second, null);
            Boolean value = prefValue != null ? "true".equalsIgnoreCase(prefValue) : null; //$NON-NLS-1$
            if (value == null) {
                preferenceNode = resolveDefaultPreferenceNode();
                value = preferenceNode != null ? preferenceNode.getBoolean(parameter.second, parameter.third) : parameter.third;
            }
            listener.execute(value);
        }
    }

    @Override
    public void unregistered() {
        IEclipsePreferences preferenceNode = resolvePreferenceNode();
        if (preferenceNode != null) {
            preferenceNode.removePreferenceChangeListener(this);
        }
        listener = null;
    }

    private IEclipsePreferences resolvePreferenceNode() {
        return InstanceScope.INSTANCE.getNode( parameter.first );
    }

    private IEclipsePreferences resolveDefaultPreferenceNode() {
        return DefaultScope.INSTANCE.getNode( parameter.first );
    }

}
