/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.request.combinations;

import java.util.HashMap;
import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;

/**
 * Functions that create new read requests by combining simpler ones.
 * @author Hannu Niemist�
 */
public class Combinators {	
		
	// ------------------------------------------------------------------------
	
	private static class Objects implements MultiRead<Resource> {
		Resource subject;		
		Resource relation;
		public Objects(Resource subject, Resource relation) {
			this.subject = subject;
			this.relation = relation;
		}
		@Override
		public void perform(ReadGraph graph,
				SyncMultiProcedure<Resource> callback)
				throws DatabaseException {
			for(Resource object : graph.getObjects(subject, relation)) {
				callback.execute(graph, object);
			}
		}		
	    @Override
	    public int hashCode() {
	        return subject.hashCode() + 31 * relation.hashCode();
	    }
	    @Override
	    public boolean equals(Object object) {
	        if (this == object) return true;
	        else if (object == null || getClass() != object.getClass()) return false;
	        Objects other = (Objects)object;
	        return subject.equals(other.subject) && relation.equals(other.relation);
	    }
	}
	
	/**
	 * Returns a multi read request that reads the objects of given subject and relation. 
	 */
	public static MultiRead<Resource> objects(Resource subject, Resource relation) {
		return new Objects(subject, relation);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Relation implements ParametrizedMultiRead<Resource, Resource> {
		Resource relation;
		public Relation(Resource relation) {
			this.relation = relation;
		}
		@Override
		public MultiRead<Resource> get(Resource subject) {
			return objects(subject, relation);
		}
	    @Override
	    public int hashCode() {
	        return getClass().hashCode() + 31 * relation.hashCode();
	    }
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Relation other = (Relation)obj;
			return relation.equals(other.relation);
		}
	}
	
	/**
	 * Returns a function <code>subject -> objects(subject, relation)</code>. 
	 */
	public static ParametrizedMultiRead<Resource,Resource> relation(Resource relation) {
		return new Relation(relation);
	}
	
	// ------------------------------------------------------------------------
	
	public static class SynchronizationProcedure<T> implements Procedure<T> {
		T result;
		DatabaseException exception;
		boolean ready = false;
		@Override
		public synchronized void exception(Throwable t) {
			this.exception = 
				t instanceof DatabaseException ? (DatabaseException)t : new DatabaseException(t);
			ready = true;
			notify();
		}
		@Override
		public synchronized void execute(T result) {
			this.result = result;
			ready = true;		
			notify();
		}		
		public synchronized T getResult() throws DatabaseException {
			if(!ready) {
				try {
					wait();
				} catch (InterruptedException e) {
					throw new DatabaseException(e);
				}	
			}
			if(exception != null)
				throw exception;
			return result;
		}
	}
	
	private static class PossibleObject implements Read<Resource> {
		Resource subject;		
		Resource relation;
		public PossibleObject(Resource subject, Resource relation) {
			this.subject = subject;
			this.relation = relation;
		}
		@Override
		public Resource perform(ReadGraph graph) throws DatabaseException {
			return graph.getPossibleObject(subject, relation);
		}	
	    @Override
	    public int hashCode() {
	        return subject.hashCode() + 31 * relation.hashCode();
	    }
	    @Override
	    public boolean equals(Object object) {
	        if (this == object) return true;
	        else if (object == null || getClass() != object.getClass()) return false;
	        PossibleObject other = (PossibleObject)object;
	        return subject.equals(other.subject) && relation.equals(other.relation);
	    }
	}
	
	/**
	 * Returns a read request that reads an object possibly connected to the subject by the relation.
	 */
	public static Read<Resource> possibleObject(Resource subject, Resource relation) {
		return new PossibleObject(subject, relation);
	}
	
	// ------------------------------------------------------------------------
	
	private static class PartialFunction implements ParametrizedRead<Resource, Resource> {
		Resource relation;
		public PartialFunction(Resource relation) {
			this.relation = relation;
		}
		@Override
		public Read<Resource> get(Resource subject) {
			return possibleObject(subject, relation);
		}
	    @Override
	    public int hashCode() {
	        return getClass().hashCode() + 31 * relation.hashCode();
	    }
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			PartialFunction other = (PartialFunction)obj;
			return relation.equals(other.relation);
		}
	}
	
	/**
	 * Returns a function <code>subject -> possibleObject(subject, relation)</code>. 
	 */
	public static ParametrizedRead<Resource,Resource> partialFunction(Resource relation) {
		return new PartialFunction(relation);
	}
	
	// ------------------------------------------------------------------------
	
	private static class SingleObject implements Read<Resource> {
		Resource subject;		
		Resource relation;
		public SingleObject(Resource subject, Resource relation) {
			this.subject = subject;
			this.relation = relation;
		}
		@Override
		public Resource perform(ReadGraph graph) throws DatabaseException {
			return graph.getSingleObject(subject, relation);
		}	
	    @Override
	    public int hashCode() {
	        return subject.hashCode() + 31 * relation.hashCode();
	    }
	    @Override
	    public boolean equals(Object object) {
	        if (this == object) return true;
	        else if (object == null || getClass() != object.getClass()) return false;
	        SingleObject other = (SingleObject)object;
	        return subject.equals(other.subject) && relation.equals(other.relation);
	    }
	}
	
	/**
	 * Returns a read request that reads an object connected to the subject by the relation.
	 */
	public static Read<Resource> singleObject(Resource subject, Resource relation) {
		return new SingleObject(subject, relation);
	}
	
	// ------------------------------------------------------------------------
	
	private static class CompleteFunction implements ParametrizedRead<Resource, Resource> {
		Resource relation;
		public CompleteFunction(Resource relation) {
			this.relation = relation;
		}
		@Override
		public Read<Resource> get(Resource subject) {
			return singleObject(subject, relation);
		}
	    @Override
	    public int hashCode() {
	        return getClass().hashCode() + 31 * relation.hashCode();
	    }
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			CompleteFunction other = (CompleteFunction)obj;
			return relation.equals(other.relation);
		}
	}
	
	/**
	 * Returns a function <code>subject -> singleObject(subject, relation)</code>. 
	 */
	public static ParametrizedRead<Resource,Resource> completeFunction(Resource relation) {
		return new CompleteFunction(relation);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Compose1<X, Y> implements Read<Y> {
		final ParametrizedRead<X, Y> f;
		final Read<X> g;		
		public Compose1(ParametrizedRead<X, Y> f, Read<X> g) {
			this.f = f;
			this.g = g;
		}
		@Override
		public Y perform(ReadGraph graph) throws DatabaseException {
			return graph.syncRequest(f.get(graph.syncRequest(g)));
		}		
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose1<?, ?> other = (Compose1<?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y> Read<Y> compose(ParametrizedRead<X, Y> f, Read<X> g) {
		return new Compose1<X, Y>(f, g);
	}

	// ------------------------------------------------------------------------
	
	private static class Compose2<X, Y, Z> implements ParametrizedRead<X, Z> {
		final ParametrizedRead<Y, Z> f;
		final ParametrizedRead<X, Y> g;		
		public Compose2(ParametrizedRead<Y, Z> f, ParametrizedRead<X, Y> g) {
			this.f = f;
			this.g = g;
		}
		public Read<Z> get(X x) {
			return compose(f, g.get(x));
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose2<?, ?, ?> other = (Compose2<?, ?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y, Z> ParametrizedRead<X, Z> compose(ParametrizedRead<Y, Z> f, ParametrizedRead<X, Y> g) {
		return new Compose2<X, Y, Z>(f, g);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Compose3<X, Y> implements MultiRead<Y> {
		final ParametrizedRead<X, Y> f;
		final MultiRead<X> g;		
		public Compose3(ParametrizedRead<X, Y> f, MultiRead<X> g) {
			this.f = f;
			this.g = g;
		}
		public void perform(ReadGraph graph, final SyncMultiProcedure<Y> callback)	throws DatabaseException {
		    try {
				for(X x : graph.syncRequest(g))
					callback.execute(graph, graph.syncRequest(f.get(x)));
				callback.finished(graph);
			} catch(DatabaseException e) {
				callback.exception(graph, e);
			}
			/*// Not sure if this is correct
		   graph.syncRequest(g, new SyncMultiProcedure<X>() {
				@Override
				public void exception(ReadGraph graph, Throwable throwable)
				throws DatabaseException {
					callback.exception(graph, throwable);					
				}
				@Override
				public void execute(ReadGraph graph, X x)
				throws DatabaseException {
					callback.execute(graph, graph.syncRequest(f.get(x)));
				}
				@Override
				public void finished(ReadGraph graph)
				throws DatabaseException {
				}
			});
			callback.finished(graph);*/
		}					
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose3<?, ?> other = (Compose3<?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y> MultiRead<Y> compose(ParametrizedRead<X, Y> f, MultiRead<X> g) {
		return new Compose3<X, Y>(f, g);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Compose4<X, Y, Z> implements ParametrizedMultiRead<X, Z> {
		final ParametrizedRead<Y, Z> f;
		final ParametrizedMultiRead<X, Y> g;		
		public Compose4(ParametrizedRead<Y, Z> f, ParametrizedMultiRead<X, Y> g) {
			this.f = f;
			this.g = g;
		}
		public MultiRead<Z> get(X x) {
			return compose(f, g.get(x));
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose4<?, ?, ?> other = (Compose4<?, ?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y, Z> ParametrizedMultiRead<X, Z> compose(ParametrizedRead<Y, Z> f, ParametrizedMultiRead<X, Y> g) {
		return new Compose4<X, Y, Z>(f, g);
	}

	// ------------------------------------------------------------------------
	
	private static class Compose5<X, Y> implements MultiRead<Y> {
		final ParametrizedMultiRead<X, Y> f;
		final Read<X> g;		
		public Compose5(ParametrizedMultiRead<X, Y> f, Read<X> g) {
			this.f = f;
			this.g = g;
		}
		@Override
		public void perform(ReadGraph graph, SyncMultiProcedure<Y> callback)
				throws DatabaseException {
			graph.syncRequest(f.get(graph.syncRequest(g)), callback);
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose5<?, ?> other = (Compose5<?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y> MultiRead<Y> compose(ParametrizedMultiRead<X, Y> f, Read<X> g) {
		return new Compose5<X, Y>(f, g);
	}

	// ------------------------------------------------------------------------
	
	private static class Compose6<X, Y, Z> implements ParametrizedMultiRead<X, Z> {
		final ParametrizedMultiRead<Y, Z> f;
		final ParametrizedRead<X, Y> g;		
		public Compose6(ParametrizedMultiRead<Y, Z> f, ParametrizedRead<X, Y> g) {
			this.f = f;
			this.g = g;
		}
		public MultiRead<Z> get(X x) {
			return compose(f, g.get(x));
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose6<?, ?, ?> other = (Compose6<?, ?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y, Z> ParametrizedMultiRead<X, Z> compose(ParametrizedMultiRead<Y, Z> f, ParametrizedRead<X, Y> g) {
		return new Compose6<X, Y, Z>(f, g);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Compose7<X, Y> implements MultiRead<Y> {
		final ParametrizedMultiRead<X, Y> f;
		final MultiRead<X> g;		
		public Compose7(ParametrizedMultiRead<X, Y> f, MultiRead<X> g) {
			this.f = f;
			this.g = g;
		}
		public void perform(ReadGraph graph, final SyncMultiProcedure<Y> callback)	throws DatabaseException {
		    try {
				for(X x : graph.syncRequest(g))
					for(Y y : graph.syncRequest(f.get(x)))
						callback.execute(graph, y);
				callback.finished(graph);
			} catch(DatabaseException e) {
				callback.exception(graph, e);
			}
			/*// Not correct because inner syncRequest calls callback.finished 
		    graph.syncRequest(g, new SyncMultiProcedure<X>() {
				@Override
				public void exception(ReadGraph graph, Throwable throwable)
				throws DatabaseException {
					callback.exception(graph, throwable);					
				}
				@Override
				public void execute(ReadGraph graph, X x) throws DatabaseException {
					graph.syncRequest(f.get(x), callback);
				}
				@Override
				public void finished(ReadGraph graph)
				throws DatabaseException {
				}
			});
			callback.finished(graph);
			*/
		}					
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose7<?, ?> other = (Compose7<?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y> MultiRead<Y> compose(ParametrizedMultiRead<X, Y> f, MultiRead<X> g) {
		return new Compose7<X, Y>(f, g);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Compose8<X, Y, Z> implements ParametrizedMultiRead<X, Z> {
		final ParametrizedMultiRead<Y, Z> f;
		final ParametrizedMultiRead<X, Y> g;		
		public Compose8(ParametrizedMultiRead<Y, Z> f, ParametrizedMultiRead<X, Y> g) {
			this.f = f;
			this.g = g;
		}
		public MultiRead<Z> get(X x) {
			return compose(f, g.get(x));
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (f.hashCode() + 31 * g.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Compose8<?, ?, ?> other = (Compose8<?, ?, ?>)obj;
			return f.equals(other.f) && g.equals(other.g);
		}
	}
	
	public static <X, Y, Z> ParametrizedMultiRead<X, Z> compose(ParametrizedMultiRead<Y, Z> f, ParametrizedMultiRead<X, Y> g) {
		return new Compose8<X, Y, Z>(f, g);
	}	
	
	// ------------------------------------------------------------------------

	private static class Index<K, V> implements Read<Map<K, V>> {
		final MultiRead<V> values;
		final ParametrizedRead<V, K> keyOfValue;
		public Index(MultiRead<V> values, ParametrizedRead<V, K> keyOfValue) {
			this.values = values;
			this.keyOfValue = keyOfValue;
		}
		@Override
		public Map<K, V> perform(ReadGraph graph) throws DatabaseException {
			HashMap<K, V> result = new HashMap<K, V>();
			for(V value : graph.syncRequest(values))
				result.put(graph.syncRequest(keyOfValue.get(value)), value);
			return result;
		}
		@Override
		public int hashCode() {
			return getClass().hashCode() + 31 * (values.hashCode() + 31 * keyOfValue.hashCode());
		}
		@Override
		public boolean equals(Object obj) {
			if(obj == this) return true;
			if(obj == null || obj.getClass() != getClass())	return false;
			Index<?, ?> other = (Index<?, ?>)obj;
			return values.equals(other.values) && keyOfValue.equals(other.keyOfValue);
		}	
	}
	
	public static <K, V> Read<Map<K, V>> index(MultiRead<V> values, ParametrizedRead<V, K> keyOfValue) {
		return new Index<K, V>(values, keyOfValue);
	}
	
	// ------------------------------------------------------------------------

	private static class Constant<T> implements Read<T> {
		T value;
		public Constant(T value) {
			this.value = value;
		}
		@Override
		public T perform(ReadGraph graph) throws DatabaseException {
			return value;
		}
		@Override
		public int hashCode() {
			return value == null ? 0 : value.hashCode();
		}
		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null || getClass() != obj.getClass())
				return false;
			Constant<?> other = (Constant<?>) obj;
			return value == null ? other.value == null : value.equals(other.value);
		}		
	}
	
	public static <T> Read<T> constant(T value) {
		return new Constant<T>(value);
	}
	
	// ------------------------------------------------------------------------

	private static class Singleton<T> implements MultiRead<T> {
		T value;
		public Singleton(T value) {
			this.value = value;
		}
		@Override
		public void perform(ReadGraph graph, SyncMultiProcedure<T> callback)
				throws DatabaseException {
			callback.execute(graph, value);
			callback.finished(graph);
		}
		@Override
		public int hashCode() {
			return value.hashCode();
		}
		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null || getClass() != obj.getClass())
				return false;
			Singleton<?> other = (Singleton<?>) obj;
			return value.equals(other.value);
		}	
	}
	
	public static <T> MultiRead<T> singleton(T value) {
		return new Singleton<T>(value);
	}
	
	// ------------------------------------------------------------------------
	
	private static class Name extends ResourceRead<String> {
		
		public Name(Resource resource) {
			super(resource);
		}
		
		@Override
		public String perform(ReadGraph graph) throws DatabaseException {
	    	Layer0 L0 = Layer0.getInstance(graph);
	    	return graph.getRelatedValue(resource, L0.HasName);
		}
		
	}	
	public static Read<String> name(Resource resource) {
		return new Name(resource);
	}
	
	// ------------------------------------------------------------------------
	
	public static final ParametrizedRead<Resource, String> NAME = new ParametrizedRead<Resource, String>() {

		@Override
		public Read<String> get(Resource resource) {
			return name(resource);
		}
		
	};
	
	// ------------------------------------------------------------------------
	
}
