/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.migration;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.layer0.Layer0;

public class OrderedSetToListMigrationStep implements MigrationStep {
    
    final ArrayList<OrderedSetToListMapping> rules;
    
    private class OrderedSetToListMapping {
        public Resource entityType, orderedSet, orderedSetRelation, list, listRelation;
    }
    
    public OrderedSetToListMigrationStep(ReadGraph graph, Resource step) throws DatabaseException {
        rules = new ArrayList<OrderedSetToListMapping>();
        Layer0 L0 = Layer0.getInstance(graph);
        for(Resource change : ListUtils.toList(graph, step)) {
            String entityType = graph.getPossibleRelatedValue(change, L0.OrderedSetToListMigrationStep_entityType, Bindings.STRING);
            String orderedSet = graph.getPossibleRelatedValue(change, L0.OrderedSetToListMigrationStep_orderedSetType, Bindings.STRING);
            String orderedSetRelation = graph.getPossibleRelatedValue(change, L0.OrderedSetToListMigrationStep_orderedSetRelation, Bindings.STRING);
            String list = graph.getPossibleRelatedValue(change, L0.OrderedSetToListMigrationStep_listType, Bindings.STRING);
            String listRelation = graph.getPossibleRelatedValue(change, L0.OrderedSetToListMigrationStep_listRelation, Bindings.STRING);
            
            if(orderedSet != null && orderedSetRelation != null && list != null && listRelation != null) {
                OrderedSetToListMapping mapping = new OrderedSetToListMapping();
                mapping.entityType = graph.getResource(entityType);
                mapping.orderedSet = graph.getResource(orderedSet);
                mapping.orderedSetRelation = graph.getResource(orderedSetRelation);
                mapping.list = graph.getResource(list);
                mapping.listRelation = graph.getResource(listRelation);
                rules.add(mapping);
            }
        }
    }

    @Override
    public void applyTo(IProgressMonitor monitor, Session session, MigrationState state) throws DatabaseException {
        final Resource resource = MigrationUtils.getResource(monitor, session, state);
        
        if(resource != null) {
            session.syncRequest(new WriteRequest() {
                
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    for(OrderedSetToListMapping mapping : rules) {
                        Instances entityFinder = graph.adapt(mapping.entityType, Instances.class);
                        for(Resource entity : entityFinder.find(graph, resource)) {
                            Resource orderedSet = graph.syncRequest(new PossibleObjectWithType(entity, mapping.orderedSetRelation, mapping.orderedSet));
                            if(orderedSet != null) {
                                Collection<Resource> list = OrderedSetUtils.toList(graph, orderedSet);
                                graph.deny(entity, mapping.orderedSetRelation, orderedSet);
                                graph.claim(entity, mapping.listRelation, ListUtils.create(graph, mapping.list, list));
                            }
                        }

                    }                    
                }
            });
        }
        
    }


}
