/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1239
 *******************************************************************************/
package org.simantics.db.layer0.bindings;

import org.simantics.databoard.binding.LongBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.LongType;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.internal.SimanticsInternal;
import org.simantics.db.service.SerialisationSupport;

/**
 * This class binds Resource to LongType
 * 
 * @author toni.kalajainen
 */
public class ResourceBinding extends LongBinding {

    public static final LongType RESOURCE_TYPE;

    /**
     * Create resource binding.
     * 
     * @param session or null
     */
    public ResourceBinding() {
        super(RESOURCE_TYPE);
    }

    /**
     * @return current {@link SerialisationSupport}
     * @throws BindingException
     *             if not able to return a current {@link SerialisationSupport}
     */
    private SerialisationSupport getCurrentSupport() throws BindingException {
        Session s = SimanticsInternal.peekSession();
        if ( s == null ) {
            throw new BindingException("Cannot instantiate Resource without an alive database Session.");
        }
        SerialisationSupport support = s.peekService(SerialisationSupport.class);
        if ( support == null ) {
            throw new BindingException("Cannot instantiate Resource without an alive database Session.");
        }
        return support;
    }

    private static long checkId(long v) throws BindingException {
        if (v == 0) throw new BindingException("resource ID cannot be 0");
        return v;
    }

    @Override
    public Object create(long value) throws BindingException {
        // NOTE: this implementation is intentionally only one that returns null
        // instead of throwing BindingException for historical reasons. 
        if (value == 0) return null;
        SerialisationSupport support = getCurrentSupport();
        try {
            return support.getResource(value);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public Object create(Long value) throws BindingException {
        long v = checkId(value.longValue());
        SerialisationSupport support = getCurrentSupport();
        try {
            return support.getResource(v);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public Object create(Number value) throws BindingException {
        long v = checkId(value.longValue());
        SerialisationSupport support = getCurrentSupport();
        try {
            return support.getResource(v);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public Object create(String value) throws BindingException {
        try {
            long v = Long.parseLong(value);
            checkId(v);
            SerialisationSupport support = getCurrentSupport();
            return support.getResource(v);
        } catch (NumberFormatException e) {
            throw new BindingException(e);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public Long getValue(Object o) throws BindingException {
        SerialisationSupport support = getCurrentSupport();
        // NOTE: r.getResourceId() is unsafe for this purpose, it will just return 0 if it fails, thus corrupting anything serialized with this method
        try {
            return support.getRandomAccessId((Resource)o);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public long getValue_(Object o) throws BindingException {
        SerialisationSupport support = getCurrentSupport();
        // NOTE: r.getResourceId() is unsafe for this purpose, it will just return 0 if it fails, thus corrupting anything serialized with this method
        try {
            return support.getRandomAccessId((Resource)o);
        } catch (DatabaseException e) {
            throw new BindingException(e);
        }
    }

    @Override
    public void setValue(Object obj, Number value) throws BindingException {
        throw new BindingException("Resource is immutable");
    }

    @Override
    public void setValue(Object obj, long value) throws BindingException {
        throw new BindingException("Resource is immutable");
    }
    
    @Override
    public boolean isImmutable() {
    	return true;
    }

    @Override
    public boolean isInstance(Object obj) {
        return obj instanceof Resource;
    }
    
    static {
        RESOURCE_TYPE = new LongType();
        RESOURCE_TYPE.metadata.put("unit", "resource");
    }
}
