/*******************************************************************************
 * Copyright (c) 2007, 2026 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import java.util.Collection;

import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Remover;
import org.simantics.db.layer0.exception.CannotRemoveException;
import org.simantics.db.layer0.request.IsContainerPublished;
import org.simantics.db.service.XSupport;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Pair;

/**
 * @author Tuukka Lehtonen
 */
public class EntityRemover extends AbstractRemover {

    public EntityRemover(Resource resource) {
        super(resource);
    }

    @Override
    public void remove(WriteGraph graph) throws DatabaseException {
        remove(graph, resource, true);
    }

    public static void remove(WriteGraph graph, Resource resource) throws DatabaseException {
        remove(graph, resource, true);
    }

    public static void remove(WriteGraph graph, final Resource resource, boolean useAdapters) throws DatabaseException {

        // Safety check
        if (graph.isImmutable(resource)) {
            XSupport xs = graph.getService(XSupport.class);
            Pair<Boolean,Boolean> sm = xs.getServiceMode();
            if(Boolean.FALSE.equals(sm.first))
                throw new CannotRemoveException("Cannot remove immutable resources (" + resource + ")");
        }
        
        boolean published = graph.syncRequest(new IsContainerPublished(resource), TransientCacheAsyncListener.instance());
        if (published)
            throw new CannotRemoveException("Items in published libraries cannot be removed. Please create a new version to perform modifications.");

        Layer0 l0 = graph.l0();
        Collection<Statement> stms = graph.getStatements(resource, l0.IsWeaklyRelatedTo);

        Resource[] composedOf = null;
        int cn = 0;

        // Remove the resource to the best of our ability.
        // NOTE: this doesn't work correctly for ordered sets, which must be removed using
        // OrderedSetUtils.
        for (Statement stm : stms) {

            Resource subject = stm.getSubject();

            if (resource.equalsResource(subject)) {
                Resource predicate = stm.getPredicate();
                Resource object = stm.getObject();
                Resource inverse = graph.getPossibleInverse(predicate);
                graph.deny(subject, predicate, inverse, object);

                if (!resource.equalsResource(object) && graph.isSubrelationOf(predicate, l0.IsComposedOf)) {
                    // Only remove entities that are not explicitly
                    // defined as a part of something else when dealing with
                    // L0.HasProperty.
                    // TODO: This logic is a temporary workaround for ontology
                    // problems. It will change in the future when we change
                    // HasProperty to inherit DependsOn instead of IsComposedOf.
                    if (graph.isSubrelationOf(predicate, l0.HasProperty)) {
                        Resource partOf = graph.getPossibleObject(object, l0.PartOf);
                        if (partOf != null && !partOf.equalsResource(resource))
                            continue;
                    }

                    if (composedOf == null)
                        composedOf = new Resource[stms.size()];
                    composedOf[cn++] = object;
                }
            }
        }

        // GitLab #1189:
        // TODO: consider using hasValue here but only once it has been optimized so that
        // it does not actually read the byte[] raw data from the clusters at all, but only
        // checks if it exists on the lowest possible level.
        //if (graph.hasValue(resource))
        graph.denyValue(resource);

        if (composedOf == null)
            return;

        // Recursively remove the objects this resource is composed of.
        for (int i = 0; i < cn; i++) {
            Resource object = composedOf[i];
            Remover remover = null;
            if (useAdapters)
                remover = graph.getPossibleAdapter(object, Remover.class);

            if (remover != null)
                remover.remove(graph);
            else
                remove(graph, object, useAdapters);
        }
    }

}
