/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;

import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

public final class DirectSuperRelations extends UnaryQuery<IntProcedure> {

	public DirectSuperRelations(int resource) {
		super(resource);
	}

	@Override
	public final void removeEntry(QueryProcessor provider) {
		provider.cache.remove(this);
	}

	private static class Ints {

		private TIntHashSet set = null;
		public int single = 0;

		public boolean add(int val) {
			if(single == val) return false;
			if(single == 0) {
				single = val;
				return true;
			}
			if(set == null) set = new TIntHashSet(4);
			return set.add(val);
		}

		public int size() {

			if(single == 0) return 0;
			if(set == null) return 1;
			return set.size() + 1;

		}

		public void forEach(TIntProcedure proc) {
			if(single > 0) proc.execute(single);
			if(set != null) set.forEach(proc);
		}

	}

	public Object compute(final ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {

		QueryProcessor processor = graph.processor;

		processor.querySupport.ensureLoaded(graph, id);

		int single = processor.querySupport.getSingleSuperrelation(id);
		if(single > 0) {
			procedure.execute(graph, single);
			procedure.finished(graph);
			return single;
		}

		final int subrelationOf = processor.getSubrelationOf();

		final IntSet result = new IntSet(processor.querySupport);

		final class DirectProcedure extends Ints implements IntProcedure, TIntProcedure {
			@Override
			final public boolean execute(int r) {
				result.add(r);
				return true;
			}
			@Override
			final public void execute(ReadGraphImpl graph, int r) {
				if(single == 0) {
					single = r;
					return;
				}
				add(r);
			}
			@Override
			public void finished(ReadGraphImpl graph) {
			}
			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				throw new Error("Errors are not supported.", t);
			}

		}

		final DirectProcedure directProc = new DirectProcedure();

		processor.querySupport.getObjects(graph, id, subrelationOf, directProc);

		int size = directProc.size();

		if(size == 0) {

			procedure.finished(graph);

		} else if (size == 1) {

			procedure.execute(graph, directProc.single);
			procedure.finished(graph);

		} else {

			directProc.forEach(new TIntProcedure() {

				@Override
				public boolean execute(int arg0) {

					try {
						procedure.execute(graph, arg0);
					} catch (DatabaseException e) {
						Logger.defaultLogError(e);
					}
					return true;

				}

			});

			procedure.finished(graph);

		}
		
		return getResult();


	}

	@Override
	public String toString() {
		return "DirectSuperRelations[" + id + "]";
	}

	@Override
	public Object performFromCache(ReadGraphImpl graph, IntProcedure procedure) throws DatabaseException {

		assert(isReady());

		return compute(graph, procedure);

	}

	@Override
	public void recompute(ReadGraphImpl graph) throws DatabaseException {

		compute(graph, new IntProcedureAdapter() {

			@Override
			public void finished(ReadGraphImpl graph) {
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				new Error("Error in recompute.", t).printStackTrace();
			}

		});

	}

}
