/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl;

import java.io.InputStream;
import java.util.function.Consumer;

import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.service.ClusterUID;

public interface ClusterI {
    public enum TypeEnum {
        Small, Big
    }
    public interface Procedure {
        
    }
    public interface PredicateProcedure<Context> extends Procedure {
        /**
         * @param context is whatever you want.
         * @param resourceKey is either subject, predicate or object resource key.
         * @return true if you want to break loop.
         */
        boolean execute(Context context, int resourceKey, int objectIndex);
    }
    public interface ObjectProcedure<Context> extends Procedure {
        /**
         * @param context is whatever you want.
         * @param resourceKey is either subject, predicate or object resource key.
         * @return true if you want to break loop.
         */
        boolean execute(Context context, int resourceKey) throws DatabaseException;
        //boolean found();

    }

    /**
     * Resource is complete if it has exactly one IIS statement. If resource has
     * multiple IIS statements it's complete reference is index to complete
     * table (and not zero).
     */
    enum CompleteTypeEnum {
        NotComplete(0),
        InstanceOf(1),
        Inherits(2),
        SubrelationOf(3);
        public static final CompleteTypeEnum make(int value) {
            switch (value) {
            default:
                throw new Error("Internal error. Illegal complete type=" + value + ".");
            case 0:
                assert(0 == ClusterI.CompleteTypeEnum.NotComplete.getValue());
                return ClusterI.CompleteTypeEnum.NotComplete;
            case 1:
                assert(1 == ClusterI.CompleteTypeEnum.InstanceOf.getValue());
                return ClusterI.CompleteTypeEnum.InstanceOf;
            case 2:
                assert(2 == ClusterI.CompleteTypeEnum.Inherits.getValue());
                return ClusterI.CompleteTypeEnum.Inherits;
            case 3:
                assert(3 == ClusterI.CompleteTypeEnum.SubrelationOf.getValue());
                return ClusterI.CompleteTypeEnum.SubrelationOf;
            }
        }
        public final byte getValue() {
            return value;
        }
        private CompleteTypeEnum(int value) {
            this.value = (byte)value;
        }
        private byte value;
    }

    enum ClusterTypeEnum {
        SMALL, BIG, WRITEONLY
    }

    public CompleteTypeEnum getCompleteType(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public int getCompleteObjectKey(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public boolean isComplete(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public <Context> boolean forPredicates(int resourceKey, PredicateProcedure<Context> procedure, Context context, ClusterSupport support)
    throws DatabaseException;

    public <Context> boolean forObjects(int resourceKey, int predicateKey, int objectIndex, ObjectProcedure<Context> procedure, Context context, ClusterSupport support)
    throws DatabaseException;

    public <Context> boolean forObjects(int resourceKey, int predicateKey, ObjectProcedure<Context> procedure, Context context, ClusterSupport support)
    throws DatabaseException;

    public int getSingleObject(int resourceKey, int predicateKey, ClusterSupport support)
    throws DatabaseException;

    public <T> int getSingleObject(int resourceKey, ForPossibleRelatedValueProcedure<T> procedure, ClusterSupport support)
    throws DatabaseException;

    public <C, T> int getSingleObject(int resourceKey, ForPossibleRelatedValueContextProcedure<C, T> procedure, ClusterSupport support)
    throws DatabaseException;

    public void forObjects(ReadGraphImpl graph, int resourceKey, int predicateKey, SyncMultiProcedure<Resource> procedure)
    throws DatabaseException;

    public void forObjects(ReadGraphImpl graph, int resourceKey, ForEachObjectProcedure procedure)
    throws DatabaseException;

    public <C> void forObjects(ReadGraphImpl graph, int resourceKey, C context, ForEachObjectContextProcedure<C> procedure)
    throws DatabaseException;

    /**
     * @param callerThread
     * @param resourceKey
     * @param predicateKey
     * @param objectKey
     * @param support
     * @return null if relation already existed. Otherwise an old or new instance of ClusterI.
     * @throws DatabaseException
     */
    public ClusterI addRelation(int resourceKey, int predicateKey, int objectKey, ClusterSupport support)
    throws DatabaseException;

    public boolean removeRelation(int resourceKey, int predicateKey, int objectKey, ClusterSupport support)
    throws DatabaseException;

    public void denyRelation(int resourceKey, int predicateKey, int objectKey, ClusterSupport support)
    throws DatabaseException;

    /**
     * @param callerThread
     * @param resourceKey
     * @param support
     * @return null if no value
     * @throws DatabaseException
     */
    public byte[] getValue(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public InputStream getValueStream(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public boolean hasValue(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public boolean removeValue(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    /**
     * @param callerThread
     * @param resourceKey
     * @param value
     * @param support
     * @return An old or new instance of ClusterI.
     * @throws DatabaseException
     */
    public ClusterI setValue(int resourceKey, byte[] value, int length, ClusterSupport support)
    throws DatabaseException;

    public ClusterI modiValueEx(int resourceKey, long voffset, int length, byte[] value, int offset, ClusterSupport support)
    throws DatabaseException;

    public byte[] readValueEx(int resourceKey, long voffset, int length, ClusterSupport support)
    throws DatabaseException;

    public long getValueSizeEx(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public void setValueEx(int resourceKey)
    throws DatabaseException;

    public int createResource(ClusterSupport support)
    throws DatabaseException;

    public boolean hasResource(int resourceKey, ClusterSupport support)
    throws DatabaseException;

    public int getNumberOfResources(ClusterSupport support) throws DatabaseException;

    public long getCachedSize();

    public long getUsedSpace()
    throws DatabaseException;

    public boolean isEmpty();

    public void printDebugInfo(String message, ClusterSupport support)
    throws DatabaseException;

    public long getClusterId();

    public int getClusterKey();

    public boolean isWriteOnly();

    public void load()
    throws DatabaseException;

    public void load(Consumer<DatabaseException> callback);
    public void load(ClusterSupport session, Runnable callback);


    public boolean hasVirtual();
    public void markVirtual();

    public ClusterI getClusterByResourceKey(int resourceKey, ClusterSupport support);
    public boolean isLoaded();
    public void increaseReferenceCount(int amount);
    public void decreaseReferenceCount(int amount);
    public int getReferenceCount();
    public long getImportance();
    public void setImportance(long i);
    public void releaseMemory();
    public void compact();

    public boolean contains(int resource);

    public ClusterTypeEnum getType();
    public boolean getImmutable();
    public void setImmutable(boolean immutable, ClusterSupport support);
    public boolean getDeleted();
    public void setDeleted(boolean deleted, ClusterSupport support);

    public ClusterUID getClusterUID();
//    public Id getModifiedId();

}
