/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common.uri;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ProcedureBarrier;
import org.simantics.db.common.WriteBindings;
import org.simantics.db.common.procedure.adapter.AsyncMultiProcedureAdapter;
import org.simantics.db.common.request.PropertyMapOfResource;
import org.simantics.db.common.request.ResourceAsyncRead;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.layer0.Layer0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EscapedChildMapOfResource extends ResourceAsyncRead<Map<String, Resource>> {

    private static final Logger LOGGER = LoggerFactory.getLogger(EscapedChildMapOfResource.class);

	public EscapedChildMapOfResource(Resource resource) {
	    super(resource);
	}

	@Override
	public void perform(AsyncReadGraph g, final AsyncProcedure<Map<String, Resource>> procedure) {
		
		final Layer0 L0 = g.getService(Layer0.class);
		final ConcurrentHashMap<String, Resource> map = new ConcurrentHashMap<String, Resource>();
		final ProcedureBarrier<Map<String, Resource>> barrier = new ProcedureBarrier<Map<String, Resource>>(1);
		
		g.forEachObject(resource, L0.ConsistsOf, new AsyncMultiProcedureAdapter<Resource>() {

			@Override
			public void execute(AsyncReadGraph g, final Resource child) {
				barrier.incrementAndGet();

				g.forPossibleRelatedValue(child, L0.HasName, WriteBindings.STRING, new AsyncProcedure<Object>() {

					@Override
					public void execute(AsyncReadGraph g, Object name) {
						if (name != null) {
							String escapedName = URIStringUtils.escape((String)name); 
							if (map.put(escapedName, child) != null)
							    LOGGER.error("The database contains siblings with the same name " + escapedName + " (resource=" + resource.getResourceId() +").");
						}
						barrier.dec(g, procedure, map);
					}
					
					@Override
					public void exception(AsyncReadGraph graph, Throwable t) {
						barrier.dec(graph, procedure, t);
					}
					
				});
			}
			
			@Override
			public void exception(AsyncReadGraph graph, Throwable t) {
				barrier.dec(graph, procedure, t);
			}
			
			@Override
			public void finished(AsyncReadGraph g) {
				barrier.dec(g, procedure, map);
			}
			
		});
	}
	
}
