/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.common;

import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;

/**
 * @author Toni Kalajainen
 */
public class ResourceArray implements Iterable<Resource> {

    public static final ResourceArray[] NONE  = new ResourceArray[0];

    public static final ResourceArray   EMPTY = new ResourceArray();

    public final Resource[]             resources;

    private int                         hash;

    private boolean noNulls(Resource[] rs) {
    	for(Resource r : rs) if(r == null) return false;
    	return true;
    }
    
    private boolean noNulls(Collection<Resource> rs) {
    	for(Resource r : rs) if(r == null) return false;
    	return true;
    }

    public ResourceArray(Resource... resources) {
    	assert(noNulls(resources));
        this.resources = resources;
        this.hash = Arrays.hashCode(resources);
    }

    public ResourceArray(Collection<Resource> resources) {
    	assert(noNulls(resources));
        this.resources = resources.toArray(new Resource[resources.size()]);
        this.hash = Arrays.hashCode(this.resources);
    }

    @Override
    public int hashCode() {
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof ResourceArray))
            return false;
        ResourceArray other = (ResourceArray) obj;
        return Arrays.deepEquals(this.resources, other.resources);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (int i = 0; i < resources.length; i++) {
            if (i > 0)
                sb.append(", ");
            sb.append(resources[i].getResourceId());
        }
        sb.append("]");
        return sb.toString();
    }

    public String toString(ReadGraph g) throws DatabaseException {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (int i = 0; i < resources.length; i++) {
            if (i > 0)
                sb.append(", ");
            sb.append(NameUtils.getSafeName(g, resources[i]));
        }
        sb.append("]");
        return sb.toString();
    }

    public static ResourceArray[] toSingleElementArrays(Resource[] array) {
        ResourceArray[] result = new ResourceArray[array.length];
        for (int i = 0; i < result.length; i++)
            result[i] = new ResourceArray(array[i]);
        return result;
    }
    
    public Resource toSingleResource() throws ValidationException {
    	if(resources.length != 1) throw new ValidationException("Resource array did not contain a single resource (contained " + resources.length + ").");
    	return resources[0];
    }

    public boolean isEmpty() {
        return resources.length == 0;
    }

    public int size() {
        return resources.length;
    }

    /**
     * @param i the index of the resource to return
     * @return ith resource of the array
     * @since 1.6
     */
    public Resource get(int i) {
        return resources[i];
    }

    @Override
    public Iterator<Resource> iterator() {
        return Arrays.asList(resources).iterator();
    }

    /**
     * Returns a new ResourceArray instance that contains the resource of this
     * ResourceArray with the specified resources appended at the end.
     * 
     * @param append the resources to append at the end of the new array
     * @return a new appended ResourceArray
     */
    public ResourceArray appended(Resource... append) {
        if (append.length == 0)
            return this;
        Resource[] result = Arrays.copyOf(resources, resources.length + append.length);
        System.arraycopy(append, 0, result, resources.length, append.length);
        return new ResourceArray(result);
    }

    /**
     * Returns a new ResourceArray instance that contains the resource of this
     * ResourceArray with the resources contained in the specified ResourceArray appended at the end.
     * 
     * @param append the ResourceArray to append at the end of the new array
     * @return a new appended ResourceArray
     */
    public ResourceArray appended(ResourceArray append) {
        if (append.size() == 0)
            return this;
        Resource[] result = Arrays.copyOf(resources, resources.length + append.size());
        System.arraycopy(append.resources, 0, result, resources.length, append.size());
        return new ResourceArray(result);
    }
    
    /**
     * Returns a new ResourceArray instance that contains the resource of this
     * ResourceArray with the specified resources prepended at the beginning.
     * 
     * @param prepend the resources to prepend at the beginning of the new array
     * @return a new prepended ResourceArray
     */
    public ResourceArray prepended(Resource... prepend) {
        if (prepend.length == 0)
            return this;
        Resource[] result = Arrays.copyOf(prepend, prepend.length + resources.length);
        System.arraycopy(resources, 0, result, prepend.length, resources.length);
        return new ResourceArray(result);
    }

    /**
     * @param n
     * @return
     * @throws IllegalArgumentException
     */
    public ResourceArray removeFromBeginning(int n) throws IllegalArgumentException {
        return slice(n, resources.length);
    }

    /**
     * @param n
     * @return
     * @throws IllegalArgumentException
     */
    public ResourceArray removeFromEnd(int n) throws IllegalArgumentException {
        return slice(0, resources.length - n);
    }

    public ResourceArray slice(int start, int end) {
        return ((start == 0) && (end == resources.length)) ? this :
            new ResourceArray(Arrays.copyOfRange(resources, start, end));
    }

    public ResourceArray reversed() {
        if (resources.length < 2)
            return this;
        Resource[] result = new Resource[resources.length];
//        for (int i = 0, j = resources.length - 1; i < resources.length / 2; ++i, --j) {
//            result[j] = resources[i];
//            result[i] = resources[j];
//        }
        for (int i = 0; i < resources.length ; ++i) {
        	result[i] = resources[resources.length - 1 - i];
        }
        return new ResourceArray(result);
    }
    
    public Resource tail() {
    	return resources[resources.length - 1];
    }

    public Resource head() {
    	return resources[0];
    }
    
}
