/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.ui;

import java.util.ArrayList;
import java.util.List;

import org.simantics.charts.query.AddChartItem;
import org.simantics.charts.query.ChartItemDescriptor;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.modeling.subscription.SCLSubscription;

/**
 * @author Jani Simomaa <jani.simomaa@semantum.fi>
 */
public class SCLChart {

	/**
	 * Creates new chart under given model
	 * 
	 * @param targetModel resource of the target model
	 * @return resource of the chart
	 * @throws DatabaseException on database failures
	 */
	public static Resource createNewChart(Resource targetModel) throws DatabaseException {
		return NewChart.createChart(targetModel);
	}

	/**
	 * Creates new chart under given model
	 * 
	 * @param targetModel resource of the target model
	 * @param chartNameProposal requested name for chart, automatically made locally unique if not
	 * @return resource of the chart
	 * @throws DatabaseException on database failures
	 */
	public static Resource createNewChart(Resource targetModel, String chartNameProposal) throws DatabaseException {
		return NewChart.createChart(targetModel, chartNameProposal);
	}

	/**
	 * Creates new chart group under given model
	 * 
	 * @param targetModel resource of the target model
	 * @return resource of the chartGroup
	 * @throws DatabaseException on database failures
	 */
	public static Resource createNewChartGroup(Resource targetModel) throws DatabaseException {
		return NewChartGroup.createChartGroup(targetModel);
	}

	/**
	 * Creates new chart group under given model
	 * 
	 * @param targetModel resource of the target model
	 * @param chartGroupNameProposal requested name for chart group, automatically made locally unique if not
	 * @return resource of the chartGroup
	 * @throws DatabaseException on database failures
	 */
	public static Resource createNewChartGroup(Resource targetModel, String chartGroupNameProposal) throws DatabaseException {
		return NewChartGroup.createChartGroup(targetModel, chartGroupNameProposal);
	}

	/**
	 * Adds chart items to the given chart
	 * 
	 * @param chart resource of the target chart
	 * @param variable variable of the item to be created
	 * @return resource of the created chart item
	 * @throws DatabaseException on database failures
	 */
	public static Resource addChartItems(WriteGraph graph, Resource chart, Variable variable) throws DatabaseException {
		Resource model = Variables.getModel(graph, variable);
		Resource defaultSubscriptionResource = SCLSubscription.defaultSubscriptionFolder(graph, model);
		Resource subscriptionItem = SCLSubscription.addSubscriptionItems(graph, defaultSubscriptionResource, variable);

		List<ChartItemDescriptor> refs = new ArrayList<ChartItemDescriptor>(1);
		refs.add( AddChartItem.createDescriptor(graph, subscriptionItem) );

		AddChartItem ci = new AddChartItem(	chart, refs	);
		ci.perform(graph);

		return ci.chartItem;
	}

	/**
	 * links given subscription to the given chart
	 * 
	 * @param subscriptionItem resource of the subscription
	 * @param chart resource of the chart
	 * @return resource of the created chart item
	 * @throws DatabaseException on database failures
	 */
	public static Resource linkSubToChart(WriteGraph graph, Resource subscriptionItem, Resource chart) throws DatabaseException {
		List<ChartItemDescriptor> refs = new ArrayList<ChartItemDescriptor>(1);
		refs.add( AddChartItem.createDescriptor(graph, subscriptionItem) );

		AddChartItem ci = new AddChartItem(	chart, refs );
		ci.perform(graph);

		return ci.chartItem;
	}

}