package org.simantics.charts.ui;

import java.util.Collections;
import java.util.Map;

import org.simantics.browsing.ui.model.labels.LabelRule;
import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveExperiment;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.subscription.SubscriptionItemLabel;

/**
 * Used for labeling both chart plots and subscription items.
 * 
 * @author Tuukka Lehtonen
 */
public class ChartItemLabelRule implements LabelRule {

    public static final ChartItemLabelRule INSTANCE = new ChartItemLabelRule();

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

    public String getLabel_(ReadGraph graph, Resource item) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        ChartResource CHART = ChartResource.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        StringBuilder sb = new StringBuilder();

        Integer index = graph.getPossibleRelatedValue(item, CHART.Chart_Item_Index, Bindings.INTEGER);
        sb.append( index==null ? "" : index+". " );

        String subscriptionItemLabel = null;
        String unitStr = null;
        SubscriptionState subscriptionState = SubscriptionState.IGNORE;
        Resource subscriptionItem = graph.getPossibleObject(item, CHART.Chart_Item_HasSubscriptionItem);
        if (subscriptionItem != null) {
            subscriptionItemLabel = graph.getPossibleRelatedValue(subscriptionItem, L0.HasLabel, Bindings.STRING);

            Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
            RVI rvi = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_VariableId, rviBinding);

            // Create label from rvi if subscription has no label
            if (subscriptionItemLabel == null || subscriptionItemLabel.isEmpty()) {
                Variable configuration = Variables.getPossibleConfigurationContext(graph, subscriptionItem);
                if (rvi != null && configuration != null) {
                    subscriptionItemLabel = rvi.asPossibleString(graph, configuration);
                    if (subscriptionItemLabel == null)
                        subscriptionItemLabel = rvi.toPossibleString(graph, configuration);
                    if (subscriptionItemLabel != null) {
                        subscriptionItemLabel = SubscriptionItemLabel.removeVariablePrefixPath(subscriptionItemLabel);
                        subscriptionItemLabel = URIStringUtils.unescape(subscriptionItemLabel);
                    }
                }
            }

            subscriptionState = getSubscriptionState(graph, rvi, subscriptionItem);
            unitStr = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Unit, Bindings.STRING);
        }
        if (subscriptionItemLabel!=null) {
            sb.append(subscriptionItemLabel);
        } else {
            sb.append("<not associated to a subscription item>");
        }

        if ( unitStr != null && !unitStr.isEmpty() ) {
            sb.append(" [");
            sb.append(unitStr);
            sb.append("]");
        }

        if (subscriptionState == SubscriptionState.INVALID)
            sb.append(" (invalid subscription)");

        return sb.toString();
    }

    private static SubscriptionState getSubscriptionState(ReadGraph graph, RVI rvi, Resource subscriptionItem) throws DatabaseException {
        Resource model = graph.sync( new PossibleModel(subscriptionItem) );
        if (model == null)
            return SubscriptionState.IGNORE;
        Resource run = graph.sync( new PossibleActiveExperiment(model) );
        if (run == null)
            return SubscriptionState.IGNORE;
        Variable base = Variables.getPossibleVariable(graph, run);
        if (base == null)
            return SubscriptionState.IGNORE;
        Variable var = rvi.resolvePossible(graph, base);
        return var != null ? SubscriptionState.VALID :  SubscriptionState.INVALID;
    }

    @Override
    public Map<String, String> getLabel(ReadGraph graph, Object content) throws DatabaseException {
        return Collections.singletonMap(ChartItemImageRule.COLUMN_KEY, getLabel_(graph, (Resource) content));
    }

    public static enum SubscriptionState {
        IGNORE,
        INVALID,
        VALID
    }

}
