/*******************************************************************************
 * Copyright (c) 2012,2023 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #925
 *******************************************************************************/
package org.simantics.charts.ui;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.NameLabelUtil;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ColumnKeys;
import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.parser.StringEscapeUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.IsParent;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.QueryIndexUtils;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.request.ProjectModels;
import org.simantics.db.request.Read;
import org.simantics.history.csv.ColumnSeparator;
import org.simantics.history.csv.DecimalSeparator;
import org.simantics.history.csv.ExportInterpolation;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.preferences.CSVPreferences;
import org.simantics.modeling.ui.modelBrowser2.label.SubscriptionItemLabelRule;
import org.simantics.utils.datastructures.Arrays;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ISelectionUtils;
import org.simantics.utils.ui.SWTUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author Antti Villberg, Tuukka Lehtonen
 */
public class CSVExportPage extends WizardPage {

    private static final Logger LOGGER = LoggerFactory.getLogger(CSVExportPage.class);

    private static class Model {
        private static final Comparator<NamedResource> COMP = 
                (o1,o2) -> AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(
                        o1.getName(), o2.getName());

        public final NamedResource model;
        public List<NamedResource> sortedSubs = new ArrayList<>();
        public Map<Resource, NamedResource> subs = new HashMap<>();
        public Map<Resource, Resource> chartItemsToSubs = new HashMap<>();
        public Set<NamedResource> initiallySelectedSubscriptions = Collections.emptySet();

        public Model(NamedResource model) {
            this.model = model;
        }

        public void initialize(ReadGraph graph, Collection<Resource> inputSelection) throws DatabaseException {
            Layer0 L0 = Layer0.getInstance(graph);
            ModelingResources MOD = ModelingResources.getInstance(graph);
            ChartResource CHART = ChartResource.getInstance(graph);
            SubscriptionItemLabelRule rule = new SubscriptionItemLabelRule();

            Resource m = model.getResource();
            String name = graph.getPossibleRelatedValue(m, L0.HasName, Bindings.STRING);
            if (name == null)
                return;

            name = NameLabelUtil.modalName(graph, m);
            for (Resource item : QueryIndexUtils.searchByTypeShallow(graph, m, MOD.Subscription_Item)) {
                String subscriptionLabel = null;
                Resource subscription = graph.getPossibleObject(item, L0.PartOf);
                if (subscription != null)
                    subscriptionLabel = graph.getPossibleRelatedValue(subscription, L0.HasLabel, Bindings.STRING);
                String label = rule.getLabel(graph, item).get(ColumnKeys.SINGLE); 
                if (label == null)
                    continue;
                if (subscriptionLabel != null)
                    label = subscriptionLabel + "/" + label; //$NON-NLS-1$
                subs.put(item, new NamedResource(label, item));
            }
            for (Resource cItem : QueryIndexUtils.searchByTypeShallow(graph, m, CHART.Chart_Item)) {
                Resource sItem = graph.getPossibleObject(cItem, CHART.Chart_Item_HasSubscriptionItem);
                if (sItem != null && subs.containsKey(sItem))
                    chartItemsToSubs.put(cItem, sItem);
            }

            sortedSubs = subs.values().stream().sorted(COMP).collect(Collectors.toList());
            initiallySelectedSubscriptions = initiallySelectedSubscriptions(graph, inputSelection);
        }

        private Set<NamedResource> initiallySelectedSubscriptions(ReadGraph graph, Collection<Resource> inputSelection) throws DatabaseException {
            if (inputSelection == null)
                return Collections.emptySet();

            HashSet<NamedResource> result = new HashSet<>();

            for (Resource i : inputSelection) {
                for (NamedResource nr : sortedSubs)
                    if (graph.syncRequest(new IsParent(i, nr.getResource())))
                        result.add(nr);
                for (Map.Entry<Resource, Resource> cs : chartItemsToSubs.entrySet())
                    if (graph.syncRequest(new IsParent(i, cs.getKey())))
                        result.add( subs.get( cs.getValue() ) );
            }

            return result;
        }
    }

    CSVExportPlan       plan;
    CCombo              model;
    Table               item;
    Button              selectAllItems;
    SelectionAdapter    selectAllItemsListener;
    CCombo              exportLocation;

    CCombo decimalSeparator;
    CCombo columnSeparator;
    CCombo sampling;
    Group resampling;
    Text timeStep;
    Text startTime;
    Text timeStamps;
    CCombo samplingMode;
    Text singlePrecision;
    Text doublePrecision;
    Button overwrite;
    CCombo encoding;

    Collection<Resource> initialSelection;
    List<Model> models = Collections.emptyList();

    ModifyListener m = (e) -> validatePage();

    SelectionListener s = new SelectionAdapter() {
        @Override
        public void widgetSelected(SelectionEvent e) {
            validatePage();
        }
    };

    protected CSVExportPage(CSVExportPlan model) {
        super(Messages.CSVExportPage_PageTitle, Messages.CSVExportPage_PageDescription, null);
        this.plan = model;
    }

    @Override
    public void createControl(Composite parent) {
        ScrolledComposite scroller = new ScrolledComposite(parent, SWT.V_SCROLL);
        scroller.setExpandHorizontal(true);
        scroller.setExpandVertical(true);

        Composite container = new Composite(scroller, SWT.NONE);
        scroller.setContent(container);
        GridLayoutFactory.swtDefaults().spacing(20, 10).numColumns(3).applyTo(container);
        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_SelectModel);
        model = new CCombo(container, SWT.BORDER);
        {
            model.setEditable(false);
            model.setText(""); //$NON-NLS-1$
            model.setToolTipText(Messages.CSVExportPage_SelectModelTT);
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(model);
        }

        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_ExportedItems);
        item = new Table(container, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION | SWT.CHECK);
        {
            item.setToolTipText(Messages.CSVExportPage_ExportedItemsTT);
            GridDataFactory.fillDefaults().grab(true, true).span(2, 1).hint(SWT.DEFAULT, 105).applyTo(item);
        }
        item.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (e.detail == SWT.CHECK) {
                    TableItem[] selected = item.getSelection();
                    TableItem it = (TableItem) e.item;
                    boolean checkedWasSelected = Arrays.contains(selected, it);
                    if (checkedWasSelected) {
                        boolean check = it.getChecked();
                        for (TableItem i : selected)
                            i.setChecked(check);
                    }
                    int checked = countCheckedItems(item);
                    int totalItems = item.getItemCount();
                    updateSelectAll(checked > 0, checked < totalItems, false);
                    validatePage();
                }
            }
        });
        new Label(container, 0);
        selectAllItems = new Button(container, SWT.CHECK);
        {
            selectAllItems.setText(Messages.CSVExportPage_SelectAll);
            selectAllItems.setToolTipText(Messages.CSVExportPage_SelectAllTT);
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(selectAllItems);
        }
        selectAllItemsListener = new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                boolean select = selectAllItems.getSelection();
                updateSelectAll(select, false, false);
                item.setRedraw(false);
                for (TableItem it : item.getItems())
                    it.setChecked(select);
                item.setRedraw(true);

                validatePage();
            }
        };
        selectAllItems.addSelectionListener(selectAllItemsListener);

        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_OutputFile);
        exportLocation = new CCombo(container, SWT.BORDER);
        {
            exportLocation.setText(""); //$NON-NLS-1$
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(exportLocation);
            exportLocation.addModifyListener(m);
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText(Messages.CSVExportPage_Browse);
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
                    dialog.setText(Messages.CSVExportPage_ChooseOutputFile);
                    dialog.setFilterPath(new File(exportLocation.getText()).getParent());
                    dialog.setFilterExtensions(new String[] { "*.csv" }); //$NON-NLS-1$
                    dialog.setFilterNames(new String[] { Messages.CSVExportPage_CSVFilterName });
                    dialog.setOverwrite(false);
                    String file = dialog.open();
                    if (file == null)
                        return;
                    exportLocation.setText(file);
                    validatePage();
                }
            });
        }

        
        Label horizRule = new Label(container, SWT.BORDER);
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 0).grab(true, false).span(3, 1).applyTo(horizRule);

        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_DecimalSeparator);
        decimalSeparator = new CCombo(container, SWT.READ_ONLY | SWT.BORDER);
        for(DecimalSeparator s : DecimalSeparator.values())
            decimalSeparator.add(s.label);
        decimalSeparator.select(0);
        decimalSeparator.addSelectionListener(s);
        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(decimalSeparator);

        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_ColumnSeparator);
        columnSeparator = new CCombo(container, SWT.READ_ONLY | SWT.BORDER);
        for(ColumnSeparator s : ColumnSeparator.values())
            columnSeparator.add(s.label);
        columnSeparator.select(0);
        columnSeparator.addSelectionListener(s);
        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(columnSeparator);

        new Label(container, SWT.NONE).setText(Messages.CSVExportPage_Sampling);
        sampling = new CCombo(container, SWT.READ_ONLY | SWT.BORDER);
        sampling.add(Messages.CSVExportPage_RecordedSamples);
        sampling.add(Messages.CSVExportPage_Resampled);
        sampling.select(0);
        sampling.addSelectionListener(s);
        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(sampling);

        resampling = new Group(container, SWT.NONE);
		resampling.setText(Messages.CSVExportPage_ResamplingSettingsNotUsed);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(resampling);
        GridLayoutFactory.swtDefaults().numColumns(3).applyTo(resampling);

        new Label(resampling, SWT.NONE).setText(Messages.CSVExportPage_ResamplingStartTime);
        startTime = new Text(resampling, SWT.BORDER);
        startTime.addModifyListener(m);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(startTime);
        new Label(resampling, SWT.NONE).setText(Messages.CSVExportPage_Seconds);

        new Label(resampling, SWT.NONE).setText(Messages.CSVExportPage_TimeStep);
        timeStep = new Text(resampling, SWT.BORDER);
        timeStep.addModifyListener(m);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(timeStep);
        new Label(resampling, SWT.NONE).setText(Messages.CSVExportPage_Seconds);
        
        new Label(resampling, SWT.NONE).setText(Messages.CSVExportPage_SamplingMode);
        samplingMode = new CCombo(resampling, SWT.READ_ONLY | SWT.BORDER);
        samplingMode.add(Messages.CSVExportPage_SamplingMode_LinearInterpolation);
        samplingMode.add(Messages.CSVExportPage_SamplingMode_PreviousSample);
        samplingMode.select(0);
        samplingMode.addSelectionListener(s);
        GridDataFactory.fillDefaults().grab(true, false).span(2,1).applyTo(samplingMode);

        Group digits = new Group(container, SWT.NONE);
        digits.setText(Messages.CSVExportPage_SignificantDigits);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(digits);
        GridLayoutFactory.swtDefaults().numColumns(2).applyTo(digits);

        new Label(digits, SWT.NONE).setText(Messages.CSVExportPage_Timestamps);
        timeStamps = new Text(digits, SWT.BORDER);
        timeStamps.addModifyListener(m);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(timeStamps);
        
        new Label(digits, SWT.NONE).setText(Messages.CSVExportPage_SinglePrecision);
        singlePrecision = new Text(digits, SWT.BORDER);
        singlePrecision.addModifyListener(m);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(singlePrecision);

        new Label(digits, SWT.NONE).setText(Messages.CSVExportPage_DoublePrecision);
        doublePrecision = new Text(digits, SWT.BORDER);
        doublePrecision.addModifyListener(m);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(doublePrecision);

        overwrite = new Button(container, SWT.CHECK);
        overwrite.setText(Messages.CSVExportPage_Overwrite);
        overwrite.setSelection(plan.overwrite);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(overwrite);
        overwrite.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                validatePage();
            }
        });

        Label csLabel = new Label(container, SWT.NONE);
        csLabel.setText(Messages.CSVExportPage_OutputEncoding);
        encoding = new CCombo(container, SWT.BORDER);
        {
            GridDataFactory.fillDefaults().grab(false, false).span(1, 1).applyTo(encoding);
            encoding.setEditable(true);
            encoding.setToolTipText(Messages.CSVExportPage_OutputEncodingTT);

            String defaultCsn = Charset.defaultCharset().name();
            String[] csns = {
                    Messages.CSVExportPage_OutputEncoding_ExcelCompatibleUnicodeText, "x-UTF-16LE-BOM", //$NON-NLS-1$
                    Messages.CSVExportPage_OutputEncoding_UTF8, "UTF-8", //$NON-NLS-1$
                    NLS.bind(Messages.CSVExportPage_OutputEncoding_SystemDefault, defaultCsn), defaultCsn,
                    Messages.CSVExportPage_OutputEncoding_USASCII, "US-ASCII" //$NON-NLS-1$
            };
            int selectedEncoding = -1;
            for (int i = 0; i < csns.length/2; ++i) {
                encoding.add(csns[i*2]);
                encoding.setData(csns[i*2], csns[i*2+1]);
                if (csns[i*2+1].equals(plan.encoding)) {
                    encoding.select(i);
                    selectedEncoding = i;
                }
            }
            if (selectedEncoding == -1)
                encoding.select(0);

            encoding.addModifyListener(e -> validatePage());
        }

        model.addSelectionListener(s);

        initialSelection = ISelectionUtils.getPossibleKeys(plan.selection, SelectionHints.KEY_MAIN, Resource.class);
        initializeWidgetsFromPreferences();
        initializeData();

        scroller.setMinSize(container.computeSize(SWT.DEFAULT, SWT.DEFAULT));
        setControl(scroller);
        validatePage();
    }

    private String getEncoding() {
        String enc = encoding.getText();
        Object data = encoding.getData(enc);
        if (data instanceof String)
            enc = (String) data;
        try {
            // Test charset name for validity.
            Charset.forName(enc);
            return enc;
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    private void initializeWidgetsFromPreferences() {
        // Write preferences to formatter
        IPreferenceStore csvnode = new ScopedPreferenceStore( InstanceScope.INSTANCE, CSVPreferences.P_NODE );

        Double startTime =  csvnode.getDouble(CSVPreferences.P_CSV_START_TIME);
        Double timeStep =  csvnode.getDouble(CSVPreferences.P_CSV_TIME_STEP);
        String decimalSeparator = csvnode.getString(CSVPreferences.P_CSV_DECIMAL_SEPARATOR);
        String columnSeparator = StringEscapeUtils.unescape( csvnode.getString(CSVPreferences.P_CSV_COLUMN_SEPARATOR) );
        Boolean resample = csvnode.getBoolean(CSVPreferences.P_CSV_RESAMPLE);
        String samplingModePreference = csvnode.getString(CSVPreferences.P_CSV_SAMPLING_MODE);
        int timeDigits = csvnode.getInt(CSVPreferences.P_CSV_TIME_DIGITS);
        int floatDigits = csvnode.getInt(CSVPreferences.P_CSV_FLOAT_DIGITS);
        int doubleDigits = csvnode.getInt(CSVPreferences.P_CSV_DOUBLE_DIGITS);

        this.decimalSeparator.select(DecimalSeparator.fromPreference(decimalSeparator).ordinal());
        this.columnSeparator.select(ColumnSeparator.fromPreference(columnSeparator).ordinal());

        this.sampling.select(resample ? 1 : 0);
        this.samplingMode.select(ExportInterpolation.fromPreference(samplingModePreference).index());

        this.startTime.setText("" + startTime); //$NON-NLS-1$
        this.timeStep.setText("" + timeStep); //$NON-NLS-1$
        this.timeStamps.setText("" + timeDigits); //$NON-NLS-1$
        this.singlePrecision.setText("" + floatDigits); //$NON-NLS-1$
        this.doublePrecision.setText("" + doubleDigits); //$NON-NLS-1$

        for (String path : plan.recentLocations)
            exportLocation.add(path);
        if (exportLocation.getItemCount() > 0)
            exportLocation.select(0);
    }

    private void initializeData() {
        try {
            getContainer().run(true, true, monitor -> {
                try {
                    initializeModelData(monitor);
                    SWTUtils.asyncExec(model, () -> {
                        if (!model.isDisposed())
                            initializeModelAndItemSelection();
                    });
                } catch (DatabaseException e) {
                    throw new InvocationTargetException(e);
                } finally {
                    monitor.done();
                }
            });
        } catch (InvocationTargetException e) {
            setErrorMessage(e.getMessage());
            LOGGER.error("Failed to initialized model data for wizard.", e.getCause()); //$NON-NLS-1$
        } catch (InterruptedException e) {
            setErrorMessage(e.getMessage());
            LOGGER.error("Interrupted wizard model data initialization.", e); //$NON-NLS-1$
        }
    }

    private void initializeModelData(IProgressMonitor monitor) throws DatabaseException {
        models = plan.sessionContext.getSession().syncRequest(
                (Read<List<Model>>) graph -> readModelData(monitor, graph));
    }

    private List<Model> readModelData(IProgressMonitor monitor, ReadGraph graph) throws DatabaseException {
        List<Model> result = new ArrayList<>();
        Layer0 L0 = Layer0.getInstance(graph);
        Collection<Resource> models = graph.syncRequest(new ProjectModels(Simantics.getProjectResource()));
        SubMonitor mon = SubMonitor.convert(monitor, Messages.CSVExportPage_Progress_ReadingModelSubscriptions, models.size());
        for (Resource model : models) {
            String name = graph.getPossibleRelatedValue(model, L0.HasName, Bindings.STRING);
            if (name != null) {
                name = NameLabelUtil.modalName(graph, model);
                mon.subTask(name);
                Model m = new Model(new NamedResource(name, model));
                m.initialize(graph, initialSelection);
                result.add(m);
            }
            mon.worked(1);
        }
        return result;
    }

    private void initializeModelAndItemSelection() {
        boolean initialSelectionDone = false;
        for (int i = 0; i < models.size(); i++) {
            Model m = models.get(i);
            model.add(m.model.getName());
            if (!initialSelectionDone) {
                boolean hasInitialSelection = m.sortedSubs.stream().anyMatch(m.initiallySelectedSubscriptions::contains);
                if (hasInitialSelection) {
                    initializeItemSelectionForModel(m);
                    initialSelectionDone = true;
                }
            }
        }
    }

    private void initializeItemSelectionForModel(Model m) {
        int i = models.indexOf(m);
        model.select(i);
        item.removeAll();
        plan.items.clear();
        int index = 0;
        int firstIndex = -1;
        item.setRedraw(false);
        for (NamedResource nr : m.sortedSubs) {
            TableItem ti = new TableItem(item, SWT.NONE);
            ti.setText(nr.getName());
            ti.setData(nr);
            if (m.initiallySelectedSubscriptions.contains(nr)) {
                plan.items.add(nr.getResource());
                ti.setChecked(true);
                if (firstIndex == -1)
                    firstIndex = index;
            }
            index++;
        }
        item.setTopIndex(Math.max(0, firstIndex));
        item.setData(m);
        item.setRedraw(true);

        int checked = countCheckedItems(item);
        updateSelectAll(checked > 0, checked < item.getItemCount(), false);
    }

    private void updateSelectAll(boolean checked, boolean gray, boolean notify) {
        if (checked) {
            selectAllItems.setText(Messages.CSVExportPage_SelectNone);
        } else {
            selectAllItems.setText(Messages.CSVExportPage_SelectAll);
        }
        if (!notify)
            selectAllItems.removeSelectionListener(selectAllItemsListener);
        selectAllItems.setGrayed(checked && gray);
        selectAllItems.setSelection(checked);
        if (!notify)
            selectAllItems.addSelectionListener(selectAllItemsListener);
    }

    protected int countCheckedItems(Table table) {
        int ret = 0;
        for (TableItem item : table.getItems())
            ret += item.getChecked() ? 1 : 0;
        return ret;
    }

    Integer validInteger(String s) {
        try {
            return Integer.parseInt(s);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    Double validDouble(String s) {
        try {
            return Double.parseDouble(s);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    Model getModel(String name) {
        for (Model m : models)
            if (m.model.getName().equals(name))
                return m;
        return null;
    }

    private void setText(Group g, String text) {
        if (!g.getText().equals(text))
            g.setText(text);
    }

    void validatePage() {
        boolean resample = sampling.getText().equals(Messages.CSVExportPage_Resampled); 
        if (resample) {
            setText(resampling, Messages.CSVExportPage_ResamplingSettings);
            timeStep.setEnabled(true);
            startTime.setEnabled(true);
            samplingMode.setEnabled(true);
        } else {
            setText(resampling, Messages.CSVExportPage_ResamplingSettingsNotUsed);
            timeStep.setEnabled(false);
            startTime.setEnabled(false);
            samplingMode.setEnabled(false);
        }

        String selectedModel = model.getText();
        Model m = getModel(selectedModel);
        if (m != null) {
            Model existing = (Model) item.getData();
            if (!m.equals(existing)) {
                item.setRedraw(false);
                item.removeAll();
                for (NamedResource sub : m.sortedSubs) {
                    TableItem ti = new TableItem(item, SWT.NONE);
                    ti.setText(sub.getName());
                    ti.setData(sub);
                    ti.setChecked(m.initiallySelectedSubscriptions.contains(sub));
                }
                item.setData(m);
                item.setRedraw(true);
            }

            plan.items = java.util.Arrays.stream(item.getItems())
                    .filter(TableItem::getChecked)
                    .map(ti -> ((NamedResource) ti.getData()).getResource())
                    .collect(Collectors.toSet());
        }

        Double validStartTime = validDouble(startTime.getText());
        Double validStepSize = validDouble(timeStep.getText());

        if (resample) {

            if (validStartTime == null) {
                setErrorMessage(Messages.CSVExportPage_Error_StartTimeMustBeNumber);
                setPageComplete(false);
                return;
            }

            if (validStepSize == null) {
                setErrorMessage(Messages.CSVExportPage_Error_StepSizeMustBeNumber);
                setPageComplete(false);
                return;
            }
            if (validStepSize <= 0) {
                setErrorMessage(Messages.CSVExportPage_Error_PositiveStepSize);
                setPageComplete(false);
                return;
            }

        } else {

            if (plan.items.size() > 1) {
                setErrorMessage(Messages.CSVExportPage_Error_RecordedSamplesOnlyForSingleItem);
                setPageComplete(false);
                return;
            }

        }

        if (item.getItemCount() == 0) {
            setErrorMessage(Messages.CSVExportPage_Error_NoSubscriptionItems);
            setPageComplete(false);
            return;
        }

        if (plan.items.isEmpty()) {
            setErrorMessage(Messages.CSVExportPage_Error_NoItemsSelected);
            setPageComplete(false);
            return;
        }

        String exportLoc = exportLocation.getText();
        if (exportLoc.isEmpty()) {
            setErrorMessage(Messages.CSVExportPage_Error_SelectOutputFile);
            setPageComplete(false);
            return;
        }
        File file = new File(exportLoc);
        if (file.isDirectory()) {
            setErrorMessage(Messages.CSVExportPage_Error_OutputFileIsDir);
            setPageComplete(false);
            return;
        }
        File parent = file.getParentFile();
        if (parent == null || !parent.isDirectory()) {
            setErrorMessage(Messages.CSVExportPage_Error_OutputDirDoesNotExist);
            setPageComplete(false);
            return;
        }

        plan.columnSeparator = ColumnSeparator.fromIndex(columnSeparator.getSelectionIndex());
        plan.decimalSeparator = DecimalSeparator.fromIndex(decimalSeparator.getSelectionIndex());
        if (plan.columnSeparator.preference.equals(plan.decimalSeparator.preference)) {
            setErrorMessage(Messages.CSVExportPage_Error_DecimalAndColumnSeparatorCannotBeSame);
            setPageComplete(false);
            return;
        }

        Integer validTimeDigits = validInteger(timeStamps.getText());

        if (validTimeDigits == null) {
            setErrorMessage(Messages.CSVExportPage_Error_TimeDigitsInteger);
            setPageComplete(false);
            return;
        }

        Integer validSinglePrecision = validInteger(singlePrecision.getText());
        if (validSinglePrecision == null) {
            setErrorMessage(Messages.CSVExportPage_Error_SinglePrecisionInteger);
            setPageComplete(false);
            return;
        }

        Integer validDoublePrecision = validInteger(doublePrecision.getText());
        if (validDoublePrecision == null) {
            setErrorMessage(Messages.CSVExportPage_Error_DoublePrecisionInteger);
            setPageComplete(false);
            return;
        }

        String encoding = getEncoding();
        if (encoding == null) {
            setErrorMessage(Messages.CSVExportPage_Error_InvalidOutputEncoding);
            setPageComplete(false);
            return;
        }

        plan.exportLocation = file;
        plan.overwrite = overwrite.getSelection();

        plan.startTime = validStartTime;
        plan.timeStep = validStepSize;

        plan.resample = sampling.getSelectionIndex() == 1;
        plan.samplingMode = ExportInterpolation.fromIndex(samplingMode.getSelectionIndex());

        plan.timeDigits = validTimeDigits;
        plan.floatDigits = validSinglePrecision;
        plan.doubleDigits = validDoublePrecision;

        plan.encoding = encoding;

        setErrorMessage(null);
        setMessage(NLS.bind(Messages.CSVExportPage_PressFinishToExport, + plan.items.size()));
        setPageComplete(true);
    }

}
