/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.query;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.subscription.FindSubscription;
import org.simantics.modeling.subscription.NewSubscription;
import org.simantics.trend.configuration.Scale;
import org.simantics.trend.impl.Plot;

public class ChartAndSubscriptionItemWriteAction extends WriteRequest {

	ChartAndSubscriptionItemData data;
	Resource chartItem;
	
	public ChartAndSubscriptionItemWriteAction(Resource chartItem, ChartAndSubscriptionItemData data)
	{
		this.data = data;
		this.chartItem = chartItem;
	}
	
	@Override
	public void perform(WriteGraph g) throws DatabaseException {
		// 0. Initialization
    	Layer0 L0 = Layer0.getInstance(g);
    	G2DResource G2D = G2DResource.getInstance(g);
    	ModelingResources MOD = ModelingResources.getInstance(g);
    	ChartResource CHART = ChartResource.getInstance(g);
        Resource chart = g.getPossibleObject(chartItem, L0.PartOf);
        Resource model = g.syncRequest(new PossibleModel( chart ));
        Resource subscriptionItem = g.getPossibleObject(chartItem, CHART.Chart_Item_HasSubscriptionItem);
        Resource subscription = subscriptionItem==null ? null : g.getPossibleObject(subscriptionItem, L0.PartOf);
		
		/// 1. Write Chart data
        // Renderer
        g.claim(chartItem, CHART.Chart_Item_Renderer, data.binaryMode ? CHART.Renderer_Binary : CHART.Renderer_Analog);

        // Drawmode
        Resource rdm = AddChartItem.getDrawmode( g, data.drawmode );
        if ( rdm != null ) {
            g.deny(chartItem, CHART.Chart_Item_DrawMode);
        	g.claim(chartItem, CHART.Chart_Item_DrawMode, rdm);
        }
        
        // Scale
    	if ( data.scale != null ) {
            Resource r = data.scale instanceof Scale.Auto ? CHART.ScaleMode_AutoScale : CHART.ScaleMode_ManualScale;
            g.deny(chartItem, CHART.Chart_Item_ScaleMode);
            g.claim(chartItem, CHART.Chart_Item_ScaleMode, r);
    	}
        if (data.min!=null) g.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Min, data.min, Bindings.DOUBLE);
        if (data.max!=null) g.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Max, data.max, Bindings.DOUBLE);

        // Visual aspects
        g.denyValue(chartItem, G2D.HasStrokeWidth);
        if (data.strokeWidth != null && data.strokeWidth.floatValue() != Plot.TREND_LINE_STROKE.getLineWidth())
            g.claimLiteral(chartItem, G2D.HasStrokeWidth, L0.Float, data.strokeWidth, Bindings.FLOAT);
        g.denyValue(chartItem, G2D.HasColor);
        if (data.color != null)
            g.claimLiteral(chartItem, G2D.HasColor, G2D.Color, data.color, Bindings.FLOAT_ARRAY);

        /// 2. Write Subscription item data
        if ( data.hasSubscriptionItem && subscriptionItem!=null )
        {
            String subscriptionLabel = g.getPossibleRelatedValue(subscription, L0.HasLabel);
            if ( data.subscription != null && !data.subscription.equals( subscriptionLabel ) ) {
            	// Wrong subscription group
            	// Find subscription 
            	Resource newSubscription = g.sync( new FindSubscription(model, null, data.subscription) );
            	
            	// Create subscription if did not exist
            	if (newSubscription == null) {
            		NewSubscription ns = new NewSubscription(model, data.subscription);
            		g.sync( ns );
            		newSubscription = ns.subscription;
            	}
            	
            	// Move item to new subscription
                g.deny(subscription, L0.ConsistsOf, subscriptionItem);
                g.claim(newSubscription, L0.ConsistsOf, subscriptionItem);
            }
            
            if (data.label!=null) g.claimLiteral(subscriptionItem, L0.HasLabel, data.label, Bindings.STRING);
            if (data.mutableCollectionSettings) {
                if (data.interval!=null) g.claimLiteral(subscriptionItem, MOD.Subscription_Item_SamplingInterval, data.interval, Bindings.DOUBLE);
                if (data.deadband!=null) g.claimLiteral(subscriptionItem, MOD.Subscription_Item_Deadband, data.deadband, Bindings.DOUBLE);
            }
            if (data.unit!=null) g.claimLiteral(subscriptionItem, MOD.Subscription_Item_Unit, data.unit, Bindings.STRING);
           	if (data.gain!=null) g.claimLiteral(subscriptionItem, MOD.Subscription_Item_Gain, data.gain, Bindings.DOUBLE);
           	if (data.bias!=null) g.claimLiteral(subscriptionItem, MOD.Subscription_Item_Bias, data.bias, Bindings.DOUBLE);
        	
        }
		
	}

}
