/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.simantics.charts.commandlog.NewChartItemCommand;
import org.simantics.charts.ontology.ChartResource;
import org.simantics.charts.preference.ChartPreferences;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.WriteResult;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.trend.configuration.Scale;
import org.simantics.trend.configuration.TrendItem.DrawMode;
import org.simantics.trend.configuration.TrendItem.Renderer;
import org.simantics.utils.commandlog.Commands;

/**
 * @author Tuukka Lehtonen
 */
public class AddChartItem extends WriteResultRequest<Collection<Resource>> {

    Resource chart;
    Collection<ChartItemDescriptor> items;
    public Resource chartItem;

    public AddChartItem(Resource chart, Collection<ChartItemDescriptor> items) {
        this.chart = chart;
        this.items = items;
    }

    @Override
    public Collection<Resource> perform(WriteGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        ChartResource CHART = ChartResource.getInstance(graph);

        int nextIndex = graph.sync( new NextChartItemIndexQuery(chart) );
        
        List<Resource> result = new ArrayList<Resource>();
        
        nextReference:
            for (ChartItemDescriptor desc : items) {
                if (desc.subscriptionItem == null)
                    continue;

                // Ensure plot is not already in the chart
                for (Resource chartItem : graph.syncRequest(Queries.objectsWithType(chart, L0.ConsistsOf, CHART.Chart_Item))) {
                    Resource subscriptionItem = graph.getPossibleObject(chartItem, CHART.Chart_Item_HasSubscriptionItem);
                    if (desc.subscriptionItem.equals(subscriptionItem)) {
                    	// Update fields
                    	if ( desc.index!=null ) {
                            graph.claimLiteral(chartItem, CHART.Chart_Item_Index, desc.index);
                            if (desc.min!=null) graph.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Min, desc.min, Bindings.DOUBLE);
                            if (desc.max!=null) graph.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Max, desc.max, Bindings.DOUBLE);
                            if (desc.renderer!=null) graph.claim(chartItem, CHART.Chart_Item_Renderer, getRenderer(graph, desc.renderer));
                            if (desc.subscriptionItem!=null) graph.claim(chartItem, CHART.Chart_Item_HasSubscriptionItem, desc.subscriptionItem);
                            
                            Resource rdm = getDrawmode( graph, desc.drawMode );
                            if ( rdm != null ) graph.claim(chartItem, CHART.Chart_Item_DrawMode, rdm);
                    	}
                        continue nextReference;
                    }
                }

                // Create & Link new plot
                Resource chartItem = graph.newResource();
                graph.claim(chartItem, L0.InstanceOf, null, CHART.Chart_Item);
                graph.claim(chart, L0.ConsistsOf, chartItem);
                graph.claimLiteral(chartItem, L0.HasName, desc.name, Bindings.STRING);
                graph.claim(chartItem, CHART.Chart_Item_HasSubscriptionItem, desc.subscriptionItem);
                graph.claim(chartItem, CHART.Chart_Item_Renderer, getRenderer(graph, desc.renderer));
                int index = desc.index!=null ? desc.index : nextIndex++;
                graph.claimLiteral(chartItem, CHART.Chart_Item_Index, index);
                if (desc.strokeWidth != null)
                    graph.claimLiteral(chartItem, G2DResource.getInstance(graph).HasStrokeWidth, desc.strokeWidth, Bindings.FLOAT);
                if (desc.color != null)
                    graph.claimLiteral(chartItem, G2DResource.getInstance(graph).HasColor, desc.color, Bindings.FLOAT_ARRAY);

                IEclipsePreferences pn = InstanceScope.INSTANCE.getNode( ChartPreferences.P_NODE );

                // Draw Mode
                {
                    String sdm = pn.get(ChartPreferences.P_DRAWMODE, ChartPreferences.DEFAULT_DRAWMODE);
                    DrawMode dm = desc.drawMode != null ? desc.drawMode : DrawMode.valueOf( sdm );
                    Resource rdm = getDrawmode( graph, dm );
                    if ( rdm != null ) {
                        graph.claim(chartItem, CHART.Chart_Item_DrawMode, rdm);
                    }
                }

                // Scale Mode
                {
                	Scale scale = desc.scale;
                	if ( scale == null ) {
                        String sdm = pn.get(ChartPreferences.P_SCALEMODE, ChartPreferences.DEFAULT_SCALEMODE);
                        if ( sdm.equals("Auto") ) {
                        	scale = new Scale.Auto();
                        } else {
                        	scale = new Scale.Manual(0, 100);
                        }                		
                	}
                	
                    if ( scale instanceof Scale.Auto ) {
                        Resource r = CHART.ScaleMode_AutoScale;
                        graph.claim(chartItem, CHART.Chart_Item_ScaleMode, r);
                    } else {
                        graph.claim(chartItem, CHART.Chart_Item_ScaleMode, CHART.ScaleMode_ManualScale);
                    }

                    if (desc.min!=null) graph.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Min, desc.min, Bindings.DOUBLE);
                    if (desc.max!=null) graph.claimLiteral(chartItem, CHART.Chart_Item_ScaleMode_Max, desc.max, Bindings.DOUBLE);
                }
                
                this.chartItem = chartItem;
                
                result.add(chartItem);
                
                if(Commands.isRecording())
                    Commands.record(graph, new NewChartItemCommand(chartItem, chart, desc.subscriptionItem));

                CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                graph.addMetadata(cm.add("Added chart item " + chartItem));
            }
        
        return result;
        
    }

    public static Resource getDrawmode(ReadGraph graph, DrawMode dm)
    {
        ChartResource CHART = ChartResource.getInstance(graph);
        if ( dm == DrawMode.Average ) return CHART.DrawMode_Average;
        if ( dm == DrawMode.Sample ) return CHART.DrawMode_Sample;
        if ( dm == DrawMode.Line ) return CHART.DrawMode_Line;
        if ( dm == DrawMode.Deviation ) return CHART.DrawMode_Deviation;
        if ( dm == DrawMode.DeviationAndAverage ) return CHART.DrawMode_DeviationAndAverage;
        if ( dm == DrawMode.DeviationAndLine ) return CHART.DrawMode_DeviationAndLine;
        if ( dm == DrawMode.DeviationAndSample ) return CHART.DrawMode_DeviationAndSample;
        return null;
    }
    
    public static Resource getRenderer(ReadGraph graph, Renderer renderer) {
        ChartResource CHART = ChartResource.getInstance(graph);
        switch (renderer) {
            case Analog:
                return CHART.Renderer_Analog;
            case Binary:
                return CHART.Renderer_Binary;
            default:
                throw new UnsupportedOperationException("invalid renderer: " + renderer);
        }
    }

    public static ChartItemDescriptor createDescriptor(ReadGraph graph, Resource subscriptionItem) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(graph);

        ChartItemDescriptor desc = new ChartItemDescriptor();
        desc.subscriptionItem = subscriptionItem;
        Datatype datatype = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Datatype, Bindings.getBindingUnchecked(Datatype.class));
        desc.renderer = datatype instanceof BooleanType ? Renderer.Binary : Renderer.Analog;

        return desc;
    }

    public static WriteResult<Collection<Resource>> addAndMoveChartItems(final Resource chart, final List<ChartItemDescriptor> references,
            final Set<Resource> movedPlots) {
        return new WriteResultRequest<Collection<Resource>>() {
            @Override
            public Collection<Resource> perform(WriteGraph graph) throws DatabaseException {
            	List<Resource> result = new ArrayList<Resource>();
                result.addAll(new AddChartItem(chart, references).perform(graph));
                Layer0 L0 = Layer0.getInstance(graph);
                for (Resource plot : movedPlots) {
                    graph.deny(plot, L0.PartOf);
                    graph.claim(chart, L0.ConsistsOf, plot);
                    result.add(plot);
                }
                return result;
            }
        };
    }

}
