/*******************************************************************************
 * Copyright (c) 2010, 2023 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #1039 
 *******************************************************************************/
package org.simantics.browsing.ui.model.nodetypes;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.WeakHashMap;

import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A node type that corresponds to some entity type.
 * @author Hannu Niemistö
 */
public class EntityNodeType implements NodeType {

    private static final Logger LOGGER = LoggerFactory.getLogger(EntityNodeType.class);

    public List<Resource> entityTypes;
    protected transient String debugName;

    private static final WeakHashMap<List<Resource>, EntityNodeType> nodeTypeCache =
        new WeakHashMap<List<Resource>, EntityNodeType>();

    private EntityNodeType(List<Resource> entityTypes) {
        this.entityTypes = entityTypes;
    }

    public static EntityNodeType create(Resource entityType) {
        return create(null, entityType);
    }

    public static EntityNodeType create(ReadGraph graph, Resource entityType) {
        List<Resource> entityTypes = Collections.singletonList(entityType);
        synchronized(nodeTypeCache) {
            EntityNodeType result = nodeTypeCache.get(entityTypes);
            if(result == null) {
                result = new EntityNodeType(entityTypes);
                try {
                    result.debugName = result.toString(graph);
                } catch (DatabaseException e) {
                    LOGGER.error("Failed to form debug name for EntityNodeType {}", result, e);
                }
                nodeTypeCache.put(entityTypes, result);
            }
            return result;
        }
    }
    
    public static EntityNodeType create(Collection<Resource> entityTypes) {
        return createInternal(null, new ArrayList<Resource>(entityTypes));
    }
    
    private static EntityNodeType createInternal(ReadGraph graph, ArrayList<Resource> entityTypes) {
        if(entityTypes.isEmpty())
            throw new IllegalArgumentException();
        else if(entityTypes.size() == 1)
            return create(graph, entityTypes.get(0));
        else {
            Collections.sort(entityTypes);
            synchronized(nodeTypeCache) {
                EntityNodeType result = nodeTypeCache.get(entityTypes);
                if(result == null) {
                    result = new EntityNodeType(entityTypes);
                    nodeTypeCache.put(entityTypes, result);
                }
                return result;
            }
        }
    }
    
    public static NodeType getNodeTypeFor(ReadGraph graph, Resource resource) throws DatabaseException {
        var types = new ArrayList<Resource>(graph.getPrincipalTypes(resource));
        if(types.isEmpty())
            return null;
        else {
            return createInternal(graph, types);
        }
    }

    public NodeType getNodeTypeOf(ReadGraph graph, Resource resource) throws DatabaseException {
        var types = new ArrayList<Resource>();
        Collection<Resource> pts = graph.getPrincipalTypes(resource);
        boolean found = false;
        loop: for(Resource t : pts) {
            for(Resource et : entityTypes) {
                if(graph.isInheritedFrom(t, et)) {
                    found = true;
                    break loop;
                }
            }
        }

        if(found)
           types.addAll(pts);

        if(types.isEmpty())
            return null;

        return createInternal(graph, types);
    }

    @Override
    public NodeContext createNodeContext(ReadGraph graph, Object content)
            throws DatabaseException {
        if(content instanceof Resource) {
            Resource resource = (Resource)content;
            NodeType nodeType = getNodeTypeOf(graph, resource);
            if(nodeType != null)
                return NodeContextBuilder.buildWithData(KEY_SEQUENCE,
                        new Object[] {content, nodeType});
        }
        return null;
    }

    @Override
    public Class<?> getContentType() {
        return Resource.class;
    }

    @Override
    public int hashCode() {
        return entityTypes.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        EntityNodeType other = (EntityNodeType) obj;
        return entityTypes.equals(other.entityTypes);
    }

    @Override
    public boolean inherits(ReadGraph graph, NodeType superType) throws DatabaseException {
        if(this == superType)
            return true;
        if (superType.getClass() != EntityNodeType.class)
            return false;
        loop:
        for(Resource st : ((EntityNodeType)superType).entityTypes) {
            for(Resource t : entityTypes)
                if(graph.isInheritedFrom(t, st))
                    continue loop;
            return false;
        }
        return true;
    }

    @Override
    public Collection<NodeType> getSuper(ReadGraph g) throws DatabaseException {
        if(entityTypes.size() == 1) {
            Layer0 l0 = Layer0.getInstance(g);
            Collection<Resource> supertypes = g.getObjects(entityTypes.get(0), l0.Inherits);
            var supernodetypes = new ArrayList<NodeType>(supertypes.size());
            for(Resource supertype : supertypes)
                supernodetypes.add(create(g, supertype));
            return supernodetypes;
        }
        else {
            var supernodetypes = new ArrayList<NodeType>(entityTypes.size());
            for(Resource t : entityTypes)
                supernodetypes.add(create(g, t));
            return supernodetypes;
        }
    }

    @Override
    public Collection<NodeType> getAll(ReadGraph g) throws DatabaseException {
        var result = new ArrayList<NodeType>(entityTypes.size());
        for(Resource t : entityTypes)
            result.add(create(g, t));
        return result;
    }

    protected String toDebugString(ReadGraph graph) throws DatabaseException {
        StringBuilder b = new StringBuilder(64);
        b.append("EntityNodeType[");
        boolean first = true;
        for(Resource t : entityTypes) {
            if(first)
                first = false;
            else
                b.append(", ");
            //b.append(graph != null ? NameUtils.getSafeName(graph, t) : t.toString());
            b.append(graph != null ? NameUtils.getURIOrSafeNameInternal(graph, t, true) : t.toString());
        }
        b.append(']');
        return b.toString();
    }

    @Override
    public String toString(ReadGraph graph) throws DatabaseException {
        if (debugName == null)
            debugName = toDebugString(graph);
        return toString();
    }

    @Override
    public String toString() {
        return debugName != null ? debugName : super.toString();
    }

    @Override
    public WorkbenchSelectionElement getWorkbenchSelectionElement(NodeContext context) {
        // TODO Auto-generated method stub
        return null;
    }

}
