/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.modifiers;

import org.simantics.browsing.ui.common.ColumnKeys;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.primitiverequest.PossibleAdapter;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.StringModifier;
import org.simantics.db.layer0.adapter.StringModifierFactory;

/**
 * @author Tuukka Lehtonen
 */
public class SingleObjectStringPropertyModifierRule implements ModifierRule {

	private Resource relation;
    private Resource propertyRelation;

    public SingleObjectStringPropertyModifierRule(ReadGraph graph, String relationURI, String propertyRelationURI) throws DatabaseException {
    	this.relation = graph.getResource(relationURI);
        this.propertyRelation = graph.getResource(propertyRelationURI);
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

    @Override
    public Modifier getModifier(ReadGraph graph, Object content,
            String columnKey) throws DatabaseException {
        if(!ColumnKeys.SINGLE.equals(columnKey))
            return null;
        if(!(content instanceof Resource))
            return null;

        Resource resource_ = (Resource)content;
        
        Resource resource = graph.getSingleObject(resource_, relation);
        
        final Statement nameStm = graph.getPossibleStatement(resource, propertyRelation);
        if (nameStm == null)
            return null;

        StringModifierFactory smf = graph.getPossibleAdapter(resource, StringModifierFactory.class);
        if (smf != null) {
            final StringModifier sm = smf.createModifier(graph, propertyRelation, nameStm.getObject());
            if (sm != null)
                return modifierFor(graph.getSession(), sm, nameStm.getObject());
        }

        return new L0StringModifier(graph.getSession(), resource, propertyRelation, nameStm.getObject());
    }

    private Modifier modifierFor(final Session session, final StringModifier sm, final Resource valueResource) {
        return new Modifier() {
            @Override
            public String getValue() {
                try {
                    return session.syncRequest(new PossibleAdapter<String>(valueResource, String.class));
                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError(e);
                    return null;
                }
            }
            @Override
            public String isValid(String label) {
                return sm.isValid(label);
            }
            @Override
            public void modify(final String label) {
                try {
                    session.syncRequest(new WriteRequest() {
                        @Override
                        public void perform(WriteGraph graph) throws DatabaseException {
                            sm.modify(graph, label);
                        }
                    });
                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError(e);
                }
            }
        };
    }

}
