/*******************************************************************************
 * Copyright (c) 2010, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.children;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;

public class CompositeChildRule implements ChildRule {

    List<ChildRule> childRules;

    public CompositeChildRule(List<ChildRule> childRules) {
        this.childRules = childRules;
    }

    private List<Resource> childRules(ReadGraph graph, Resource ordered) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        if(graph.isInstanceOf(ordered, L0.OrderedSet)) {
            return OrderedSetUtils.toList(graph, ordered);
        } else if(graph.isInstanceOf(ordered, L0.List)) {
            return ListUtils.toList(graph, ordered);
        } else {
            throw new DatabaseException("Expected list or ordered set, got " + ordered);
        }
    }
    
    public CompositeChildRule(ReadGraph graph, Resource ordered) throws DatabaseException {
        if(ordered == null) {
            childRules= Collections.emptyList();
            return;
        }
        List<Resource> childRuleResources = childRules(graph, ordered);
        childRules = new ArrayList<ChildRule>(childRuleResources.size());
        for(Resource childRuleResource : childRuleResources)
            childRules.add(graph.adapt(childRuleResource, ChildRule.class));
    }

    @Override
    public Collection<?> getChildren(ReadGraph graph, Object parent) throws DatabaseException {
        if(childRules.isEmpty())
            return Collections.emptyList();
        Collection<Object> result = Collections.singleton(parent);
        for(int i=0;i<childRules.size();++i) {
            ChildRule rule = childRules.get(i);
            ArrayList<Object> children = new ArrayList<Object>();
            for(Object r : result)
                children.addAll(rule.getChildren(graph, r));
            result = children;
        }
        return result;
    }
    
    @Override
    public Collection<?> getParents(ReadGraph graph, Object child) throws DatabaseException {
        Collection<Object> result = Collections.singleton(child);
        for(int i=childRules.size()-1;i>=0;--i) {
            ChildRule rule = childRules.get(i);
            ArrayList<Object> parents = new ArrayList<Object>();
            for(Object r : result)
                parents.addAll(rule.getParents(graph, r));
            result = parents;
        }
        return result;
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        if(childRules.isEmpty())
           return true; 
        return childRules.get(0).isCompatible(contentType);
    }

}
