/*******************************************************************************
 * Copyright (c) 2010, 2021 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - #793 - support actionDefinitionId
 *******************************************************************************/
package org.simantics.browsing.ui.model.actions;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.resource.ImageDescriptor;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeTypeMultiMap;
import org.simantics.browsing.ui.model.tests.Test;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.layer0.adapter.ActionFactory2;
import org.simantics.db.layer0.adapter.PrioritizedFactory;
import org.simantics.utils.ui.action.PriorityAction;
import org.simantics.viewpoint.ontology.ViewpointResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ActionContribution {

    private static final Logger LOGGER = LoggerFactory.getLogger(ActionContribution.class);

    final Resource rule;
    final NodeType nodeType;
    final Test isVisible;
    final Test isEnabled;
    final String label;
    final ActionFactory actionFactory;
    final IActionCategory category;
    final ImageDescriptor image;
    final String actionDefinitionId;

    public ActionContribution(Resource rule, NodeType nodeType, Test isVisible,
            Test isEnabled, String label,
            ActionFactory actionFactory, IActionCategory category,
            ImageDescriptor image, String actionDefinitionId) {
    	this.rule = rule;
        this.nodeType = nodeType;
        this.isVisible = isVisible;
        this.isEnabled = isEnabled;
        this.label = label;
        this.actionFactory = actionFactory;
        this.category = category;
        this.image = image;
        this.actionDefinitionId = actionDefinitionId;
    }
    
    public static void load(ReadGraph g, Resource r, NodeTypeMultiMap<ActionContribution> contributions) throws DatabaseException {
        ViewpointResource vr = ViewpointResource.getInstance(g);

        Resource isVisibleResource = g.getPossibleObject(r, vr.ActionContribution_IsVisibleIf);
        Test isVisible = isVisibleResource == null ? null : g.adapt(isVisibleResource, Test.class);

        Resource isEnabledResource = g.getPossibleObject(r, vr.ActionContribution_IsEnabledIf);
        Test isEnabled = isEnabledResource == null ? null : g.adapt(isEnabledResource, Test.class);

        String label = NameUtils.getSafeLabel(g, r);

        ActionFactory actionFactory = g.adapt(g.getSingleObject(r, vr.ActionContribution_HasAction), ActionFactory.class);

        Resource categoryResource = g.getPossibleObject(r, vr.ActionContribution_HasCategory);
        IActionCategory category = categoryResource == null ? null : g.adapt(categoryResource, IActionCategory.class);

        Resource imageResource = g.getPossibleObject(r, vr.ActionContribution_HasImage);
        ImageDescriptor image = imageResource == null ? null : g.adapt(imageResource, ImageDescriptor.class);

        String actionDefinitionId = g.getPossibleRelatedValue(r, vr.ActionContribution_HasActionDefinitionId, Bindings.STRING);

        for(Resource nodeTypeResource : g.getObjects(r, vr.ActionContribution_HasNodeType)) {
            NodeType nodeType = g.adapt(nodeTypeResource, NodeType.class);

            ActionContribution contribution = 
                new ActionContribution(r, nodeType, isVisible, isEnabled, label, actionFactory, category, image, actionDefinitionId);
            contributions.put(nodeType, contribution);
        }
    }

    static class ActionContributionAction extends PriorityAction {
    	
    	public Resource contribution;
    	public Runnable runnable;
    	
    	public ActionContributionAction(int priority, String label, Resource rule, Runnable runnable) {
    		super(priority, label);
    		this.contribution = rule;
    		this.runnable = runnable;
    	}
    	
        @Override
        public void run() {
            runnable.run();
        }
    	
    }

    public CategorizedAction getAction(ReadGraph graph, NodeContext context, Collection<NodeContext> all) {
        Object content = context.getConstant(BuiltinKeys.INPUT);
        try {
            if(isVisible == null || isVisible.test(graph, content)) {
            	Collection<Object> allContents = new ArrayList<Object>();
            	for(NodeContext ctx : all) allContents.add(ctx.getConstant(BuiltinKeys.INPUT));
            	final Runnable runnable = (actionFactory instanceof ActionFactory2) ? 
            			((ActionFactory2)actionFactory).create(allContents) : actionFactory.create(content);
            	int prio = actionFactory instanceof PrioritizedFactory pf ? pf.priority() : 0;
            	if(runnable == null) return null;
            	Action action = new ActionContributionAction(prio, label, rule, runnable);
                action.setId(runnable.getClass().getName());
                action.setEnabled(isEnabled == null || isEnabled.test(graph, content));
                action.setActionDefinitionId(actionDefinitionId);
                if(image != null)
                    action.setImageDescriptor(image);
                return new CategorizedAction(action, category);
            }
            else
                return null;
        } catch(DatabaseException e) {
            LOGGER.error("Failed to get CategorizedAction", e);
            return null;
        }
    }
}
