# Introduction

`Devs` module is a small toolkit for building discrete event models with SCL. The basic basic
building blocks are *Events*, *Signals*, *Tasks* and *random numbers*.

A model is created and simulated as
~~~
model = createModel do
    <initialization code>
simulateDuration model <simulation duration>
~~~

The model object maintains its state so it may be simulated multiple times with small durations
for example when synchronizing the execution to some other simulation.

Events and Signals are communication channels between the different parts of the model.
Events happen only at certain points of time while signal values persist and change only at discrete times. 

Consider for example an auction. We have an event
    
    bids :: Event (String, Double)
    
that triggers every time somebody calls out his bid (the String value is the name of the bidder and the Double value the prize).
We have also the signals

    price :: Signal Double
    winner :: Signal (Maybe String)
    auctionOpen :: Signal Boolean
    
that tells the current price of the item (that may be either the initial minimum bid or the highest bid), the current highest bidder
and if the auction is still open for new bids. Finally we have some auction rules that determine which bids are accepted.

    acceptedBids :: Event (String, Double)

The auctioneer controls the auction process and closes it after that have not been new accepted bids for a while. So there is a task

    closeAuction :: Task
    
The whole model can be defined and tested in the following way:

    auction () = do
        // Model parameters
        minimumBid = 100
        increment = 10
        closingTime = 3.0
        
        model = createModel do
            // Main events and signals
            bids = createEvent
            price = createSignal minimumBid
            winner = createSignal Nothing
            auctionOpen = createSignal True
            
            // Logic for accepting the bids
            acceptedBids = filterEvent (\(_,p) -> p >= currentValue price + increment && currentValue auctionOpen) bids
            
            // Logic for closing the auction
            closeAuction = createTask (updateSignal auctionOpen False)
            listenEvent acceptedBids (\_ -> schedule closeAuction (currentTime + closingTime))
            schedule closeAuction (currentTime + closingTime)
            
            // Update winner and price
            listenEvent acceptedBids (\(a,p) -> do
                updateSignal winner (Just a)
                updateSignal price p)
            
            // What really happened during the auction
            at 1.0 (sendEvent bids ("Charles", 150.0))
            at 1.2 (sendEvent bids ("Oscar", 170.0))
            at 1.6 (sendEvent bids ("Charles", 200.0))
            at 1.9 (sendEvent bids ("Oscar", 205.0))    // too small increment
            at 4.7 (sendEvent bids ("Oscar", 210.0))    // too late, auction already closed
            
            // Print the winner when the auction has been closed
            listenSignal auctionOpen (\_ -> print "Winner is \(fromJust $ currentValue winner)")
        simulateDuration model 10.0

# Definitions

## Events

::data[Event]
::value[createEvent, sendEvent, listenEvent]

::value[mapEvent, filterEvent, createEventFromSignal, mergeEvents]

## Signals

::data[Signal]
::value[createSignal, currentValue, updateSignal, listenSignal]

::value[mapSignal, lift1Signal, lift2Signal, lift3Signal, liftSignals, foldEvent]

## Controlling time

::value[currentTime]

::value[at, after]

::data[Task]
::value[createTask, schedule]

## Random numbers

::value[rndUniform, rndChoice, rndInteger, rndGaussian]

## Models

::data[DevsModel]
::value[createModel]
::value[simulateDuration]
