/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils;

import java.io.File;
import java.util.Properties;

/**
 * 
 * @author Toni Kalajainen
 */
public class LibraryPathUtils {

    // The problem with JMF is that it is (should be)
    // Located in Java CLASSPATH. For some reason eclipse java VM does not
    // Look into CLASSPATH.
    // The following code adds CLASSPATH to System enviroment java.library.path
    public static void addCLASSPATHtoEclipseVariables() {
        String CLASSPATH = System.getenv("CLASSPATH");
        if (CLASSPATH != null)
            for (String classPath : CLASSPATH.split(";")) {
                if (!classPath.toLowerCase().endsWith(".jar"))
                    addLibraryPath(classPath);
                addClassPath(classPath);
            }
        CLASSPATH = System.getenv("PATH");
        if (CLASSPATH != null)
            for (String classPath : CLASSPATH.split(";")) {
                addLibraryPath(classPath);
            }
        // On linux
        CLASSPATH = System.getenv("LD_LIBRARY_PATH");
        if (CLASSPATH != null)
            for (String classPath : CLASSPATH.split(";")) {
                addLibraryPath(classPath);
            }

        // printEnvVariables();
    }

    public static void addLibraryPath(String newLibrary) {
        if (newLibrary == null || "".equals(newLibrary))
            return;
        String java_library_path = System.getProperty("java.library.path");
        if (java_library_path != null) {
            for (String library : java_library_path.split(";"))
                if (library.equals(newLibrary))
                    return;
        }

        // System.out.println("Adding path: " + newLibrary);

        if (java_library_path == null || "".equals(java_library_path))
            java_library_path = newLibrary;
        else {
            if (!java_library_path.endsWith(";"))
                java_library_path += ";";
            java_library_path += newLibrary + ";";
        }

        System.setProperty("java.library.path", java_library_path);
    }

    public static void addClassPath(String newLibrary) {
        if (newLibrary == null || "".equals(newLibrary))
            return;
        String java_library_path = System.getProperty("java.class.path");
        if (java_library_path != null) {
            for (String library : java_library_path.split(";"))
                if (library.equals(newLibrary))
                    return;
        }

        // System.out.println("Adding cp: " + newLibrary);

        if (java_library_path == null || "".equals(java_library_path))
            java_library_path = newLibrary;
        else {
            if (!java_library_path.endsWith(";"))
                java_library_path += ";";
            java_library_path += newLibrary + ";";
        }

        System.setProperty("java.class.path", java_library_path);
    }

    public static void printEnvVariables() {
        // Add env variable CLASSPATH
        Properties props = System.getProperties();
        for (Object key : props.keySet()) {
            Object value = props.get(key);
            System.out.println(key + "=" + value);
        }

        System.out.println("\n\nEnv:");
        for (Object key : System.getenv().keySet()) {
            Object value = System.getenv().get(key);
            System.out.println(key + "=" + value);
        }
    }

    /**
     * Find location of a jar library
     * 
     * @param library name of the library
     * @return full location of the library
     */
    public static String findLibrary(String library) {
        String CLASSPATH = System.getenv("CLASSPATH");

        if (CLASSPATH == null)
            return null;
        // 1st search case sensitive
        for (String classPath : CLASSPATH.split(";"))
            if (classPath.endsWith(library))
                return classPath;

        // Widen search for insensitive
        for (String classPath : CLASSPATH.split(";"))
            if (classPath.toLowerCase().endsWith(library.toLowerCase()))
                return classPath;
        return null;
    }       
    
    /**
     * This method searches for CLASSPATH and converts libraries there into
     * separate key,value pairs in system properties. The key will be 
     * name of the library and value will be it's location in file system.
     *
     * This method acts as a helper tool for loading external libraries into 
     * the visibility of bundle class loader.
     * Call:
     * LibraryPathUtils.createSystemPropertiesFromClassPathLibraries(); 
     * 
     * This call should be invoked at the very beginning of application. 
     * This way external libraries can by dynamicly loaded at runtime. 
     * The way to do this, is to add "external:$library.jar$" into plugin 
     * settings runtime/classpath, 
     * in Manifest.mf it is "Bundle-ClassPath: external:$library.jar$" 
     */
    public static void createSystemPropertiesFromClassPathLibraries() {
        String CLASSPATH = System.getenv("CLASSPATH");

        if (CLASSPATH == null)
            return;
        
        // 1st search case sensitive
        for (String classPath : CLASSPATH.split(";")) {
            //System.out.println(classPath);
            if (!classPath.toLowerCase().endsWith(".jar")) continue;
            File lib = new File(classPath);
            if (!lib.exists()) {
                //System.out.println("Warning library \""+classPath+"\" does not exist yet it exists in CLASSPATH.");
                continue;
            } else {
            }
            String key = lib.getName();
            String value = lib.getAbsolutePath();
            System.setProperty(key, value);
            //System.out.println(key+"="+value);
        }
    }

}
