/*******************************************************************************
 * Copyright (c) 2010, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.visuals;

import org.simantics.browsing.ui.model.InvalidContribution;
import org.simantics.browsing.ui.model.check.CheckedStateContribution;
import org.simantics.browsing.ui.model.check.CheckedStateRule;
import org.simantics.browsing.ui.model.imagedecorators.ImageDecorationContribution;
import org.simantics.browsing.ui.model.imagedecorators.ImageDecorationRule;
import org.simantics.browsing.ui.model.images.ImageContribution;
import org.simantics.browsing.ui.model.images.ImageRule;
import org.simantics.browsing.ui.model.labeldecorators.LabelDecorationContribution;
import org.simantics.browsing.ui.model.labeldecorators.LabelDecorationRule;
import org.simantics.browsing.ui.model.labels.LabelContribution;
import org.simantics.browsing.ui.model.labels.LabelRule;
import org.simantics.browsing.ui.model.modifiers.ModifierContribution;
import org.simantics.browsing.ui.model.modifiers.ModifierRule;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.browsing.ui.model.nodetypes.OrderedNodeTypeMultiMap;
import org.simantics.browsing.ui.model.sorters.SorterContribution;
import org.simantics.browsing.ui.model.sorters.SorterRule;
import org.simantics.browsing.ui.model.tests.Test;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.AdaptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.viewpoint.ontology.ViewpointResource;

/**
 * Produces labels for nodes of given node type.
 * @author Hannu Niemistö
 */
public class VisualsContribution implements Comparable<VisualsContribution> {
    protected NodeType nodeType;
    protected Test test;
    protected double priority;
    
    public VisualsContribution(NodeType nodeType, Test test, double priority) throws InvalidContribution {
        if(test != null && !test.isCompatible(
                nodeType.getContentType()
                ))
            throw new InvalidContribution("Test is not compatible with the content type.");
        this.nodeType = nodeType;
        this.test = test;
        this.priority = priority;
    }
    
    public static void load(ReadGraph g, Resource visualsContributionResource,
            OrderedNodeTypeMultiMap<LabelContribution> labelContributions,
            OrderedNodeTypeMultiMap<ImageContribution> imageContributions,
            OrderedNodeTypeMultiMap<CheckedStateContribution> checkedStateContributions,
            OrderedNodeTypeMultiMap<LabelDecorationContribution> labelDecorationContributions,
            OrderedNodeTypeMultiMap<ImageDecorationContribution> imageDecorationContributions,
            OrderedNodeTypeMultiMap<ModifierContribution> modifierContributions,
            OrderedNodeTypeMultiMap<SorterContribution> sorterContributions,
            OrderedNodeTypeMultiMap<FlatNodeContribution> flatNodeContributions) 
    throws DatabaseException, InvalidContribution {
        ViewpointResource vr = ViewpointResource.getInstance(g);
        
        Resource testResource = g.getPossibleObject(visualsContributionResource, vr.VisualsContribution_HasCondition);
        Test test = testResource == null ? null : g.adapt(testResource, Test.class);
        
        Double mpriority = g.getPossibleRelatedValue(visualsContributionResource, vr.VisualsContribution_HasPriority);
        double priority = mpriority == null ? 0.0 : mpriority.doubleValue();         
        
        for(Resource nodeTypeResource : g.getObjects(visualsContributionResource, vr.VisualsContribution_HasNodeType)) {
            NodeType nodeType = g.adapt(nodeTypeResource, NodeType.class);

            for(Resource ruleResource : g.getObjects(visualsContributionResource, vr.VisualsContribution_HasRule)) {
                if(ruleResource.equals(vr.FlatNodeRule)) 
                    flatNodeContributions.put(nodeType, FlatNodeContribution.INSTANCE);
                else {          
                    VisualsRule rule;
                    try {
                        rule = g.adapt(ruleResource, VisualsRule.class);
                    } catch(AdaptionException e) {
                        e.printStackTrace();
                        continue;
                    }                    
                    try {
                        // Note: the rule may be an instance of multiple interfaces
                        if(rule instanceof LabelRule)
                            labelContributions.put(nodeType, new LabelContribution(nodeType, test, (LabelRule)rule, priority));
                        if(rule instanceof ModifierRule)
                            modifierContributions.put(nodeType, new ModifierContribution(nodeType, test, (ModifierRule)rule, priority));
                        if(rule instanceof ImageRule)
                            imageContributions.put(nodeType, new ImageContribution(nodeType, test, (ImageRule)rule, priority));
                        if(rule instanceof CheckedStateRule)
                            checkedStateContributions.put(nodeType, new CheckedStateContribution(nodeType, test, (CheckedStateRule)rule, priority));
                        if(rule instanceof LabelDecorationRule)
                            labelDecorationContributions.put(nodeType, new LabelDecorationContribution(nodeType, test, (LabelDecorationRule)rule, priority));
                        if(rule instanceof ImageDecorationRule)
                            imageDecorationContributions.put(nodeType, new ImageDecorationContribution(nodeType, test, (ImageDecorationRule)rule, priority));
                        if(rule instanceof SorterRule)
                            sorterContributions.put(nodeType, new SorterContribution(nodeType, test, (SorterRule)rule, priority));
                    } catch(InvalidContribution e) {
                        e.printStackTrace();
                        continue;
                    }
                }
            }
        }
    }   
    
    public NodeType getNodeType() {
        return nodeType;
    }

    @Override
    public int compareTo(VisualsContribution o) {
        return Double.compare(o.priority, priority);
    }
}
