/*******************************************************************************
 * Copyright (c) 2010, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.labeldecorators;

import org.eclipse.jface.resource.ColorDescriptor;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;
import org.simantics.browsing.ui.content.LabelDecorator;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.viewpoint.ontology.ViewpointResource;

/**
 * Constant label decoration rule decorates the label in a fixed way.
 * @author Hannu Niemistö
 */
public class ConstantLabelDecorationRule extends AbstractLabelDecorator implements LabelDecorationRule {

    String prefix;
    String postfix;
    ColorDescriptor foregroundColor;
    ColorDescriptor backgroundColor;
    int style;
    
    public ConstantLabelDecorationRule(String format,
            ColorDescriptor foregroundColor, ColorDescriptor backgroundColor,
            String style) {
        if(format != null) {
            String[] split = format.split("%s", 2);
            if(split.length == 1) {
                prefix = "";
                postfix = split[0];
            }
            else {
                prefix = split[0];
                postfix = split[1];
            }
        }
        this.foregroundColor = foregroundColor;
        this.backgroundColor = backgroundColor;
        this.style = SWT.NORMAL;
        if(style != null)
            for(char c : style.toCharArray())
                switch(c) {
                case 'B':
                case 'b':
                    this.style |= SWT.BOLD;
                    break;
                case 'I':
                case 'i':
                    this.style |= SWT.ITALIC;
                    break;
                default:
                    System.err.println("Invalid character '" + c + "' in style string. Only B and I recognized.");
                }
    }

    private static ColorDescriptor getPossibleRelatedColor(ReadGraph g, Resource subject, Resource predicate) throws DatabaseException {
        Resource value = g.getPossibleObject(subject, predicate);
        if(value == null)
            return null;
        RGB rgb = g.adapt(value, RGB.class);
        return ColorDescriptor.createFrom(rgb);
    }
    
    public static ConstantLabelDecorationRule create(ReadGraph g, Resource r) throws DatabaseException {
        ViewpointResource vr = ViewpointResource.getInstance(g);        
        String format = g.getPossibleRelatedValue(r, vr.ConstantLabelDecorationRule_HasFormat, Bindings.STRING);

        ColorDescriptor foregroundColor = getPossibleRelatedColor(g, r, vr.ConstantLabelDecorationRule_HasForegroundColor);
        ColorDescriptor backgroundColor = getPossibleRelatedColor(g, r, vr.ConstantLabelDecorationRule_HasBackgroundColor);
        
        String style = g.getPossibleRelatedValue(r, vr.ConstantLabelDecorationRule_HasStyle, Bindings.STRING);
        
        return new ConstantLabelDecorationRule(
                format, 
                foregroundColor, 
                backgroundColor,
                style
        );
    }
    
    @Override
    public boolean isCompatible(Class<?> contentType) {
        return true;
    }

    @Override
    public LabelDecorator getLabelDecorator(ReadGraph graph, Object content)
            throws DatabaseException {
        return this;
    }
    
    @Override
    public String decorateLabel(String label, String column, int itemIndex) {
        if(postfix == null)
            return label;
        else
            return prefix + label + postfix;
    }
    
    @SuppressWarnings("unchecked")
    public <Color> Color decorateForeground(Color color, String column, int itemIndex) {
        if(foregroundColor == null)
            return color;
        else
            return (Color)foregroundColor;
    }

    @SuppressWarnings("unchecked")
    public <Color> Color decorateBackground(Color color, String column, int itemIndex) {
        if(backgroundColor == null)
            return color;
        else
            return (Color)backgroundColor;
    }
    
    @SuppressWarnings("unchecked")
    public <Font> Font decorateFont(Font font, String column, int itemIndex) {
        if(style == 0)
            return font;
        else {
            FontDescriptor desc = (FontDescriptor)font;
            return (Font)desc.withStyle(style);
        }
    }
}
