/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.browsing.ui.Tester;
import org.simantics.browsing.ui.common.EvaluatorData.Evaluator;
import org.simantics.browsing.ui.common.EvaluatorData.EvaluatorTree;
import org.simantics.browsing.ui.content.CheckedStateFactory;
import org.simantics.browsing.ui.content.ComparableContextFactory;
import org.simantics.browsing.ui.content.ImageDecoratorFactory;
import org.simantics.browsing.ui.content.ImagerFactory;
import org.simantics.browsing.ui.content.LabelDecoratorFactory;
import org.simantics.browsing.ui.content.LabelerFactory;
import org.simantics.browsing.ui.content.ViewpointFactory;

/**
 * @author Tuukka Lehtonen
 */
public class EvaluatorImpl implements Evaluator {

    private final EvaluatorTree<ViewpointFactory>      viewpointTree           = new EvaluatorTreeImpl<ViewpointFactory>("Viewpoint", Testers.PASS);
    private final EvaluatorTree<LabelerFactory>        labelerTree             = new EvaluatorTreeImpl<LabelerFactory>("Labeler", Testers.PASS);
    private final EvaluatorTree<CheckedStateFactory>     checkStateTree             = new EvaluatorTreeImpl<CheckedStateFactory>("Check state", Testers.PASS);
    private final EvaluatorTree<LabelDecoratorFactory> labelDecoratorTree      = new EvaluatorTreeImpl<LabelDecoratorFactory>("Label decorator", Testers.PASS);
    private final EvaluatorTree<ImagerFactory>         imagerTree              = new EvaluatorTreeImpl<ImagerFactory>("Imager", Testers.PASS);
    private final EvaluatorTree<ImageDecoratorFactory> imageDecoratorTree              = new EvaluatorTreeImpl<ImageDecoratorFactory>("Image decorator", Testers.PASS);
    private final EvaluatorTree<ComparableContextFactory>     comparableContextTree          = new EvaluatorTreeImpl<ComparableContextFactory>("Comparable", Testers.PASS);

    @Override
    public Evaluator addViewpoint(ViewpointFactory factory, double preference) {
        viewpointTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addComparableContext(ComparableContextFactory factory, double preference) {
        comparableContextTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addLabeler(LabelerFactory factory, double preference) {
        labelerTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addCheckState(CheckedStateFactory factory, double preference) {
        checkStateTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addLabelDecorator(LabelDecoratorFactory factory, double preference) {
        labelDecoratorTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addComparator(ComparableContextFactory factory, double preference) {
        comparableContextTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addImager(ImagerFactory factory, double preference) {
        imagerTree.addFactory(factory, preference);
        return this;
    }
    @Override
    public Evaluator addImageDecorator(ImageDecoratorFactory factory, double preference) {
        imageDecoratorTree.addFactory(factory, preference);
        return this;
    }

    @Override
    public EvaluatorTree<ComparableContextFactory> getComparableContextTree() {
        return comparableContextTree;
    }

    @Override
    public EvaluatorTree<ImagerFactory> getImagerTree() {
        return imagerTree;
    }

    @Override
    public EvaluatorTree<ImageDecoratorFactory> getImageDecoratorTree() {
        return imageDecoratorTree;
    }

    @Override
    public EvaluatorTree<LabelDecoratorFactory> getLabelDecoratorTree() {
        return labelDecoratorTree;
    }

    @Override
    public EvaluatorTree<LabelerFactory> getLabelerTree() {
        return labelerTree;
    }

    @Override
    public EvaluatorTree<CheckedStateFactory> getCheckStateTree() {
        return checkStateTree;
    }

    @Override
    public EvaluatorTree<ViewpointFactory> getViewpointTree() {
        return viewpointTree;
    }

    /**
     * @param <Factory>
     */
    public class EvaluatorTreeImpl<Factory> implements EvaluatorTree<Factory> {
        private final String                             factory;
        private final Tester                             tester;
        private final Collection<Preference<Factory>>    acceptedFactories = new ArrayList<Preference<Factory>>();
        private final Collection<EvaluatorTree<Factory>> children          = new ArrayList<EvaluatorTree<Factory>>();

        public EvaluatorTreeImpl(String factory, Tester tester) {
            if (tester == null)
                throw new NullPointerException("null tester");
            this.tester = tester;
            this.factory = factory;
        }

        @Override
        public EvaluatorTree<Factory> addFactory(Factory factory, double preference) {
            if (factory == null)
                throw new NullPointerException("null factory");
            acceptedFactories.add(new Preference<Factory>(factory, preference));
            return this;
        }

        @Override
        public EvaluatorTree<Factory> addBranch(Tester tester) {
            if (tester == null)
                throw new NullPointerException("null tester");
            EvaluatorTree<Factory> result = new EvaluatorTreeImpl<Factory>(factory, tester);
            children.add(result);
            return result;
        }

        @Override
        public Collection<Preference<Factory>> getAcceptedFactories() {
            return acceptedFactories;
        }

        @Override
        public Collection<EvaluatorTree<Factory>> getChildren() {
            return children;
        }

        @Override
        public Tester getTester() {
            return tester;
        }

        @Override
        public String toString() {
            return factory + " tree [" + tester + "]";
        }

    }

    @Override
    public String toString() {
        return "Evaluator";
    }

}