/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics;

import static org.simantics.db.common.utils.Transaction.commit;
import static org.simantics.db.common.utils.Transaction.endTransaction;
import static org.simantics.db.common.utils.Transaction.readGraph;
import static org.simantics.db.common.utils.Transaction.startTransaction;
import static org.simantics.db.common.utils.Transaction.writeGraph;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ForkJoinPool;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IProduct;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.osgi.service.datalocation.Location;
import org.eclipse.osgi.service.resolver.BundleDescription;
import org.ini4j.Ini;
import org.ini4j.InvalidFileFormatException;
import org.simantics.SimanticsPlatform.OntologyRecoveryPolicy;
import org.simantics.SimanticsPlatform.RecoveryPolicy;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.Datatypes;
import org.simantics.datatypes.literal.Font;
import org.simantics.datatypes.literal.RGB;
import org.simantics.db.Driver;
import org.simantics.db.Driver.Management;
import org.simantics.db.Manager;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.SessionModel;
import org.simantics.db.UndoContext;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.StandardStoredModuleSupport;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.processor.MergingDelayedWriteProcessor;
import org.simantics.db.common.processor.MergingGraphRequestProcessor;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.common.utils.Transaction;
import org.simantics.db.exception.ClusterSetExistException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.indexing.DatabaseIndexing;
import org.simantics.db.layer0.genericrelation.DependenciesRelation;
import org.simantics.db.layer0.genericrelation.IndexException;
import org.simantics.db.layer0.genericrelation.IndexedRelations;
import org.simantics.db.layer0.request.PossibleResource;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.util.SimanticsClipboardImpl;
import org.simantics.db.layer0.util.SimanticsKeys;
import org.simantics.db.layer0.util.TGTransferableGraphSource;
import org.simantics.db.layer0.variable.VariableRepository;
import org.simantics.db.management.SessionContext;
import org.simantics.db.request.Read;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.service.LifecycleSupport.LifecycleListener;
import org.simantics.db.service.LifecycleSupport.LifecycleState;
import org.simantics.db.service.QueryControl;
import org.simantics.db.service.UndoRedoSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.db.service.XSupport;
import org.simantics.db.services.GlobalServiceInitializer;
import org.simantics.graph.db.GraphDependencyAnalyzer;
import org.simantics.graph.db.GraphDependencyAnalyzer.IU;
import org.simantics.graph.db.GraphDependencyAnalyzer.IdentityNode;
import org.simantics.graph.db.IImportAdvisor;
import org.simantics.graph.db.ImportResult;
import org.simantics.graph.db.StreamingTransferableGraphImportProcess;
import org.simantics.graph.db.TransferableGraphs;
import org.simantics.graph.diff.Diff;
import org.simantics.graph.diff.TransferableGraphDelta1;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.internal.Activator;
import org.simantics.internal.TimedSessionCache;
import org.simantics.internal.startup.StartupExtensions;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;
import org.simantics.project.IProject;
import org.simantics.project.ProjectFeatures;
import org.simantics.project.ProjectKeys;
import org.simantics.project.Projects;
import org.simantics.project.SessionDescriptor;
import org.simantics.project.exception.ProjectException;
import org.simantics.project.features.registry.GroupReference;
import org.simantics.project.management.DatabaseManagement;
import org.simantics.project.management.GraphBundle;
import org.simantics.project.management.GraphBundleEx;
import org.simantics.project.management.GraphBundleRef;
import org.simantics.project.management.PlatformUtil;
import org.simantics.project.management.ServerManager;
import org.simantics.project.management.ServerManagerFactory;
import org.simantics.project.management.WorkspaceUtil;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptions;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptionsAdvisor;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.utils.FileUtils;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.strings.EString;
import org.simantics.utils.threads.ExecutorWorker;
import org.simantics.utils.threads.ThreadUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * SimanticsPlatform performs procedures required in order to get simantics
 * workbench into operational state. This consists of the following steps:
 * <ul>
 *     <li> Asserting there is Database
 *     </li>
 *     <li> Starting Database process
 *     </li>
 *     <li> Opening a session to Database process
 *     </li>
 *     <li> Asserting required ontologies or other transferable graphs are installed in the database
 *     </li>
 *     <li> Asserting required project is installed in the database
 *     </li>
 *     <li> Asserting Simantics Features are installed in the database
 *     </li>
 *     <li> Asserting Simantics Features are installed to the project
 *     </li>
 *     <li> Shutdown: Save Session, Close session, Kill Database process
 *     </li>
 * </ul>
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class SimanticsPlatform implements LifecycleListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(SimanticsPlatform.class);
    
    /**
     * The policy is relevant when developing Simantics from Eclipse IDE.
     * It is applied when the ontology in the database of a workspace doesn't match
     * a newer ontology in the Eclipse workspace.
     */
    public static enum OntologyRecoveryPolicy { ThrowError, Merge, ReinstallDatabase, Bypass}

    /**
     * This policy dictates how the Simantics platform startup should react if
     * the started workspace is not set up properly. The alternatives are to
     * just throw an error and fail or to attempt all possible measures to fix
     * the encountered problems.
     */
    public static enum RecoveryPolicy { ThrowError, FixError }

    /** Singleton instance, started in SimanticsWorkbenchAdvisor */
    public static final SimanticsPlatform INSTANCE = new SimanticsPlatform();

    /** Set to true when the Simantics Platform is in good-and-go condition */
    public boolean running;

    /** ID of the database driver that the platform is currently using */
    private String currentDatabaseDriver;

    /** Database Session */
    public Session session;
    private Management databasebManagement;

    /** Database session context */
    public SessionContext sessionContext;

    /** Project identifier in Database */
    public String projectURI;

    /** Project name */
    public String projectName;

    /** Project resource */
    public Resource projectResource;

    /** Session specific bindings */
    public SimanticsBindings simanticsBindings;

    public Thread mainThread;

    private Thread shutdownHook = new Thread() {
        @Override
        public void run() {
            try {
                LOGGER.warn("Simantics platform was not properly shut down. Executing safety shutdown hook.");
                shutdown(null, false);
            } catch (PlatformException e) {
                LOGGER.error("Simantics Platform shutdown hook execution failed.", e);
                log.log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Simantics Platform shutdown hook execution failed.", e));
            }
        }
    };

    /**
     * The {@link IProject} activated by
     * {@link #startUp(IProgressMonitor, RecoveryPolicy, OntologyRecoveryPolicy, ServerAddress, PlatformUserAgent)}
     */
    private IProject project;

    protected ILog log;

    /**
     * Create a new simantics plaform manager in uninitialized state and
     * with default policies. <p>
     */
    public SimanticsPlatform() {
        log = Platform.getLog(Activator.getBundleContext().getBundle());
        mainThread = Thread.currentThread();
    }

    public String getApplicationClientId() {
        IProduct product = Platform.getProduct();
        if(product == null) return "noProduct";//UUID.randomUUID().toString();
        String application = product.getApplication();
        return application != null ? application : UUID.randomUUID().toString();
    }

    private SessionDescriptor setupDatabase(String databaseDriverId, IProgressMonitor progressMonitor, RecoveryPolicy workspacePolicy, PlatformUserAgent userAgent) throws PlatformException {
        if (progressMonitor == null)
            progressMonitor = new NullProgressMonitor();
        Path workspaceLocation = Platform.getLocation().toFile().toPath();
        Path dbLocation = dbLocation();
        Path dbIniPath = workspaceLocation.resolve("db.ini");
        // The driver file overrides any command line arguments to prevent
        // using the wrong driver for an existing database directory.
        ServerManager serverManager;
        try {
            Ini dbIni = loadOrCreateDatabaseIni(dbIniPath, databaseDriverId);
            databaseDriverId = dbIni.get("driver", "id");
            serverManager = ServerManagerFactory.create(databaseDriverId, dbLocation.toAbsolutePath().toString());
        } catch (DatabaseException | IOException e) {
            throw new PlatformException("Failed to initialize database ServerManager with driver " + databaseDriverId, e);
        }
        progressMonitor.beginTask("Setting up Simantics Database", 100);
        progressMonitor.setTaskName("Asserting Database is installed.");
        String msg = "Failed to initialize Simantics database.";
        try {
            // Create database
            log.log(new Status(IStatus.INFO, Activator.PLUGIN_ID, "Initializing database at " + dbLocation + " with driver " + databaseDriverId));
            progressMonitor.setTaskName("Creating database at " + dbLocation);
            databasebManagement = serverManager.getManagement(dbLocation.toFile());
            databasebManagement.create();
            currentDatabaseDriver = databaseDriverId;
            // Create layer0.
            return serverManager.createDatabase(dbLocation.toFile());
        } catch (DatabaseException e) {
            throw new PlatformException(msg, e);
        } catch (Throwable e) {
            throw new PlatformException(msg, e);
        } finally {
            progressMonitor.worked(20);
        }
    }

    public void synchronizeOntologies(IProgressMonitor progressMonitor, OntologyRecoveryPolicy ontologyPolicy, boolean requireSynchronize) throws PlatformException {

        SubMonitor monitor = SubMonitor.convert(progressMonitor, 100);

        // #955 - Old mechanism which was disabled by default for 1.56.0
        if (Boolean.parseBoolean(System.getProperty("org.simantics.platform.compileDynamicOntologies", "false"))) { //$NON-NLS-1$ $NON-NLS-2$
            monitor.setTaskName("Compile dynamic ontologies");
            PlatformUtil.compileAllDynamicOntologies();
        }

        String message = "Asserting all ontologies are installed";
        LOGGER.info(message);
        monitor.setTaskName(message);

        DatabaseManagement mgmt = new DatabaseManagement();
        Map<GraphBundleRef, GraphBundleEx> platformTGs = new HashMap<>();
        try {

            // Get a list of bundles installed into the database
            message = "find installed bundles from database";
            monitor.subTask(message);
            LOGGER.info(message);
            Map<GraphBundleRef, GraphBundleEx> installedTGs = new HashMap<>();
            for (GraphBundle b : session.syncRequest( mgmt.GraphBundleQuery )) {
                installedTGs.put(GraphBundleRef.of(b), GraphBundleEx.extend(b));
            }

            if(!requireSynchronize && installedTGs.size() > 1 && !Platform.inDevelopmentMode()) return;
//            if(installedTGs.size() > 1) return;

            // Get a list of all bundles in the platform (Bundle Context)
            message = "load all transferable graphs from platform";
            monitor.subTask(message);
            LOGGER.info(message);
            Collection<GraphBundle> tgs = PlatformUtil.getAllGraphs();
            message = "extend bundles to compile versions";
            monitor.subTask(message);
            LOGGER.info(message);
            for (GraphBundle b : tgs) {
                GraphBundleEx gbe = GraphBundleEx.extend(b);
                gbe.build();
                platformTGs.put(GraphBundleRef.of(b), gbe);
            }

            // Compile a list of TGs that need to be installed or reinstalled in the database
            message = "check bundle reinstallation demand";
            monitor.subTask(message);
            LOGGER.info(message);
            List<GraphBundleEx> installTGs = new ArrayList<>();
            // Create list of TGs to update, <newTg, oldTg>
            Map<GraphBundleEx,GraphBundleEx> reinstallTGs = new TreeMap<>();
            for (Entry<GraphBundleRef, GraphBundleEx> e : platformTGs.entrySet()) {
                GraphBundleRef key = e.getKey();
                GraphBundleEx platformBundle = e.getValue();
                GraphBundleEx existingBundle = installedTGs.get(key);
                
//                System.out.println("GraphBundleRef key=" + key.toString());
                
                if (existingBundle == null) {
                    // Bundle did not exist in the database, put it into list of bundles to install
                    installTGs.add(platformBundle);
                }
                else {
                    // Bundle exists in the database
                    boolean platformBundleIsNewer = existingBundle.getVersion().compareTo(platformBundle.getVersion())<0;
                    if (!platformBundleIsNewer)
                        continue;
                    // Check hash of transferable graph to know whether to update or not.
                    if (platformBundle.getHashcode() == existingBundle.getHashcode())
                        continue;
                    //System.out.println("Ontology hashcodes do not match: platform bundle="
                    //        + platformBundle.getVersionedId() + ", hash=" + platformBundle.getHashcode()
                    //        + "; existing bundle=" + existingBundle.getVersionedId() + ", hash=" + existingBundle.getHashcode());
                    reinstallTGs.put(platformBundle, existingBundle);
                }
            }
            // INSTALL
            // Database is missing graphs
            if (!installTGs.isEmpty() || !reinstallTGs.isEmpty()) {
                session.getService(XSupport.class).setServiceMode(true, true);

                // Throw error
                if (ontologyPolicy == OntologyRecoveryPolicy.ThrowError || ontologyPolicy == OntologyRecoveryPolicy.Bypass) {
                    StringBuilder sb = new StringBuilder("The following graphs are not installed in the database: ");
                    if (!installTGs.isEmpty()) {
                        int i = 0;
                        for (GraphBundleEx e : installTGs) {
                            if (i>0) sb.append(", ");
                            i++;
                            sb.append(e.toString());
                        }
                        sb.append(" is missing from the database.\n");
                    }
                    if (!reinstallTGs.isEmpty()) {
                        int i = 0;
                        for (Entry<GraphBundleEx, GraphBundleEx> e : reinstallTGs.entrySet()) {
                            if (i>0) sb.append(", ");
                            i++;
                            sb.append(e.getKey().toString());
                        }
                        sb.append(" Database/Platform Bundle version mismatch.\n");
                    }
                    sb.append("Hint: Use -fixErrors to install the graphs.");
                    if (ontologyPolicy == OntologyRecoveryPolicy.ThrowError)
                    	throw new PlatformException(sb.toString());
                    else
                    	log.log(new Status(IStatus.WARNING, Activator.PLUGIN_ID, sb.toString()));
                }
                
                // Reinstall database
                if (ontologyPolicy == OntologyRecoveryPolicy.ReinstallDatabase) {
                    log.log(new Status(IStatus.INFO, Activator.PLUGIN_ID, "Reinstalling the database."));
                    // TODO Install DB
                    // Stop Session
                    // Kill Process
                    // Delete Database
                    // Create Database
                    // Start Database
                    // Open Session
                    // Install TGs
                    throw new PlatformException("Reinstalling Database, NOT IMPLEMENTED");
                }

                boolean serviceModeEntered = false;

                if (ontologyPolicy == OntologyRecoveryPolicy.Merge) {
                    message = "Merging ontology changes";
                    monitor.subTask(message);
                    LOGGER.info(message);
                    // Sort missing TGs into install order
                    GraphDependencyAnalyzer<GraphBundle> analyzer = new GraphDependencyAnalyzer<GraphBundle>();
                    for(GraphBundle tg : installTGs) analyzer.addGraph(tg, tg.getGraph());
                    for(GraphBundle tg : reinstallTGs.keySet()) analyzer.addGraph(tg, tg.getGraph());
                    if(!analyzer.analyzeDependency()) {
                        Collection<Pair<GraphBundle, GraphBundle>> problems = analyzer.getConflicts();
                        StringBuilder sb = new StringBuilder();
                        for (Pair<GraphBundle, GraphBundle> problem : problems) {
                            sb.append("Conflict with "+problem.first+" and "+problem.second+".\n");
                        }
                        throw new PlatformException(sb.toString());
                    } else if(!session.syncRequest( analyzer.queryExternalDependenciesSatisfied )) {
                        Collection<IdentityNode> unsatisfiedDependencies = analyzer.getUnsatisfiedDependencies();
                        StringBuilder sb = new StringBuilder();
                        for (IdentityNode dep: unsatisfiedDependencies) {
                            sb.append("Unsatisfied Dependency "+dep+". Required by\n");
                            for(IU iu : GraphDependencyAnalyzer.toCollection(dep.getRequires())) {
                                sb.append("    " + ((GraphBundle)iu.getId()).getId() + "\n");
                            }
                        }
                        throw new PlatformException(sb.toString());
                    }

                    message = "Analyzed graph bundles";
                    monitor.subTask(message);
                    LOGGER.info(message);

                    List<GraphBundle> sortedBundles = analyzer.getSortedGraphs();
                    if(!sortedBundles.isEmpty()) {

                        session.syncRequest((Write) graph -> {
                            try {
                                graph.newClusterSet(graph.getDefaultMutableClusterSet());
                            } catch (ClusterSetExistException e) {
                                // Cluster set exist already, no problem.
                            }
                            try {
                                graph.newClusterSet(graph.getDefaultImmutableClusterSet());
                            } catch (ClusterSetExistException e) {
                                // Cluster set exist already, no problem.
                            }
                            try {
                                graph.newClusterSet(graph.getRootLibrary());
                            } catch (ClusterSetExistException e) {
                                // Cluster set exist already, no problem.
                            }
                            graph.setClusterSet4NewResource(graph.getDefaultImmutableClusterSet());
                            graph.flushCluster();
                        });

                        boolean mergedOntologies = false;

                        // Install TGs
                        for(final GraphBundle tg : sortedBundles) {

                            final IImportAdvisor advisor = new OntologyImportAdvisor(tg, mgmt);
                            final GraphBundle oldTG = reinstallTGs.get(tg);

                            boolean createImmutable = tg.getImmutable();

                            if (oldTG==null) {

                                session.getService(XSupport.class).setServiceMode(true, createImmutable);

                                // Flush all queries once to allow even immutable request results to be invalidated
                                // because ontology installation affects immutable content and queries related to
                                // immutable content would not get invalidated at all otherwise.
                                if (!serviceModeEntered) {
                                    serviceModeEntered = true;
                                    session.getService(QueryControl.class).flush();
                                }

                                // Install TG
                                log.log(new Status(IStatus.INFO, Activator.PLUGIN_ID, "Installing "+tg.toString()+" - "+tg.getName()));
                                ImportResult result = TransferableGraphs.importGraph1(session, null, new TGTransferableGraphSource(tg.getGraph()), advisor, null, false, StreamingTransferableGraphImportProcess.failOnMissingEntities(), false);
                                if (result.hasMissingExternals()) {
                                    log.log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Import of " + tg.toString() + " was missing the following external entities:\n" + EString.implode(result.missingExternals())));
                                } else {
                                    session.syncRequest((WriteOnly) graph -> {
                                        // Invalidate the points where new content was added
                                        session.getService(XSupport.class).invalidateRequests(graph, result.resolvedExternals(session));
                                    });
                                }
                            } else {
                                if(!createImmutable)
                                    continue;

                                // Merge TG
                                startTransaction(session, false);

                                TransferableGraph1 old = oldTG.getGraph();
                                if(old == null) {
                                    endTransaction();
                                    continue;
                                }
                                
                                TransferableGraphDelta1 delta = new Diff(oldTG.getGraph(), tg.getGraph()).diff();
                                final long[] oldResources = oldTG.getResourceArray();
                                boolean changes = TransferableGraphs.hasChanges(readGraph(), oldResources, delta);
                                endTransaction();
                                if (!changes) {
                                    log.log(new Status(IStatus.INFO, Activator.PLUGIN_ID, "Nothing to merge for "+tg.toString()));
                                    continue;
                                }

                                log.log(new Status(IStatus.INFO, Activator.PLUGIN_ID, "Merging new version of "+tg.toString()));

                                startTransaction(session, true);

                                //delta.print();
                                try {
                                    long[] resourceArray = TransferableGraphs.applyDelta(writeGraph(), oldResources, delta);
                                    tg.setResourceArray(resourceArray);
                                    mgmt.setGraphBundleEntry(tg);
                                    commit();
                                    mergedOntologies = true;
                                } catch (Throwable t) {
                                    throw new PlatformException(t);
                                } finally {
                                    endTransaction();
                                }

                                session.syncRequest((Write) graph -> {
                                    QueryControl qc = graph.getService(QueryControl.class);
                                    qc.flush(graph);
                                });
                        	}
                        }

                        session.syncRequest((Write) graph -> {
                            graph.setClusterSet4NewResource(graph.getDefaultImmutableClusterSet());
                            graph.flushCluster();
                        });

                        if (mergedOntologies)
                            DatabaseIndexing.deleteAllIndexes();
                    }
                }
                session.getService(XSupport.class).setServiceMode(false, false);
                if (serviceModeEntered) {
                    // Flush all queries to ensure that queries that should now
                    // be immutable are not left as mutable in the query cache.
                    session.getService(QueryControl.class).flush();
                }
            }
            message = "Ontologies synchronized";
            monitor.subTask(message);
            LOGGER.info(message);
            monitor.worked(100);
        } catch (IOException e) {
            throw new PlatformException(e);
        } catch (DatabaseException e) {
            throw new PlatformException(e);
        }

    }

    public boolean assertConfiguration(IProgressMonitor progressMonitor, RecoveryPolicy workspacePolicy) throws PlatformException {

        if (progressMonitor == null) progressMonitor = new NullProgressMonitor();

        File workspaceLocation = Platform.getLocation().toFile();

        boolean installProject = false;
        progressMonitor.setTaskName("Asserting simantics.cfg is installed");
        try {
            File propertyFile = new File(workspaceLocation, "simantics.cfg");
            Properties properties;
            try {
                properties = WorkspaceUtil.readProperties(propertyFile);
            } catch (IOException e) {
                if (workspacePolicy == RecoveryPolicy.ThrowError) throw new PlatformException("Could not load "+propertyFile);

                // Create a project and write Property file
                properties = new Properties();
                properties.setProperty("project_uri", "http://Projects/Development%20Project");
                properties.setProperty("project_name", "Development Project");
                WorkspaceUtil.writeProperties(propertyFile, properties);
                installProject |= true;
            }
            projectURI = properties.getProperty("project_uri");
            projectName = properties.getProperty("project_name");
            progressMonitor.worked(10);
        } catch (IOException e) {
            throw new PlatformException(e);
        }

        return installProject;

    }

    public boolean assertProject(IProgressMonitor progressMonitor, RecoveryPolicy workspacePolicy, boolean installProject) throws PlatformException {

        SubMonitor monitor = SubMonitor.convert(progressMonitor, 10);

        final DatabaseManagement mgmt = new DatabaseManagement();

        monitor.setTaskName("Asserting project resource exists in the database");
        try {
            projectResource = session.syncRequest(new PossibleResource(projectURI));
            if (projectResource == null) {
                // Project was not found
                if (workspacePolicy == RecoveryPolicy.ThrowError)
                    throw new PlatformException("Project Resource "+projectURI+" is not found in the database.");
                // Create empty project with no features
                try {
                    Transaction.startTransaction(session, true);
                    try {
                        // The project needs to be created mutable.
                        session.getService(XSupport.class).setServiceMode(true, false);

                        ArrayList<String> empty = new ArrayList<String>();
                        projectResource = mgmt.createProject(projectName, empty);
                        installProject |= true;

                        session.getService(XSupport.class).setServiceMode(false, false);
                        Transaction.commit();
                    } finally {
                        Transaction.endTransaction();
                    }

                    // Flush queries after changes to immutable resources
                    session.getService(QueryControl.class).flush();

                    //session.getService( LifecycleSupport.class ).save();
                } catch (DatabaseException e) {
                    throw new PlatformException("Failed to create "+projectURI, e);
                }
            }
        } catch (DatabaseException e) {
            throw new PlatformException("Failed to create "+projectURI, e);
        }
        monitor.worked(10);

        return installProject;

    }

    public void updateInstalledGroups(IProgressMonitor progressMonitor, boolean installProject) throws PlatformException {

        if (installProject)
        {
            // Attach all feature groups available in platform to created project
            progressMonitor.setTaskName("Install all features");
            Set<GroupReference> publishedFeatureGroups = ProjectFeatures.getInstallGroupsOfPublishedFeatures();
            Collection<GroupReference> groupsWithoutVersion = GroupReference.stripVersions(publishedFeatureGroups);

            try {
                session.syncRequest(
                        (Write) graph ->
                        Projects.setProjectInstalledGroups(graph, projectResource, groupsWithoutVersion));
            } catch (DatabaseException ae) {
                throw new PlatformException("Failed to install features", ae);
            }
            progressMonitor.worked(10);
        }

    }

    public void assertSessionModel(IProgressMonitor progressMonitor) throws PlatformException {

        Properties properties = session.getService(Properties.class);
        final String clientId = properties.getProperty("clientId");

        try {

            // Currently this needs to be done before data becomes available
            VirtualGraphSupport support = session.getService(VirtualGraphSupport.class);
            VirtualGraph activations = support.getWorkspacePersistent("activations");

            Resource sessionModel = session.syncRequest(new Read<Resource>() {

                @Override
                public Resource perform(ReadGraph graph) throws DatabaseException {

                    Layer0X L0X = Layer0X.getInstance(graph);
                    for(Resource sessionModel : graph.syncRequest(new ObjectsWithType(graph.getRootLibrary(), L0X.HasSession, L0X.Session))) {
                        String id = graph.getPossibleRelatedValue(sessionModel, L0X.Session_HasClientId);
                        if(id != null && id.equals(clientId)) return sessionModel;
                    }
                    return null;

                }

            }, TransientCacheAsyncListener.instance());

            if(sessionModel == null) {

                sessionModel = session.syncRequest(new WriteResultRequest<Resource>(activations) {

                    @Override
                    public Resource perform(WriteGraph graph) throws DatabaseException {
                        Layer0 L0 = Layer0.getInstance(graph);
                        Layer0X L0X = Layer0X.getInstance(graph);
                        Resource session = graph.newResource();
                        graph.claim(session, L0.InstanceOf, null, L0X.Session);
                        graph.claim(session, L0X.Session_HasUser, null, graph.getResource("http://Users/AdminUser"));
                        graph.addLiteral(session, L0X.Session_HasClientId, L0X.Session_HasClientId_Inverse, clientId, Bindings.STRING);
                        graph.claim(graph.getRootLibrary(), L0X.HasSession, session);
                        return session;
                    }
                });

            }

            session.registerService(SessionModel.class, new PlatformSessionModel(sessionModel));
        } catch (DatabaseException e) {
            throw new PlatformException(e);
        }

    }

    static class PlatformSessionModel implements SessionModel {
        private final Resource sessionModel;

        public PlatformSessionModel(Resource model) {
            this.sessionModel = model;
        }

        @Override
        public Resource getResource() {
            return sessionModel;
        }
    }

    public void resetDatabase(IProgressMonitor monitor) throws PlatformException {
        // TODO: fix this to use Path APIs
        File dbLocation = dbLocation().toFile();
        if(!dbLocation.exists())
            return;
        try { // Load driver
            Driver driver = Manager.getDriver("acorn");
            Management management = driver.getManagement(dbLocation.getAbsolutePath(), null);
            management.delete();
        } catch (DatabaseException e) {
            throw new PlatformException("Failed to remove database at " + dbLocation.getAbsolutePath(), e);
        }
        // We have created extra files to database folder which have to be deleted also.
        // This is an awful idea! Do not create extra files to database folder!
        Throwable t = null;
        for (int i=0; i<10; ++i) {
            try {
                FileUtils.deleteAll(dbLocation);
                t = null;
                break;
            } catch (IOException e) {
                // Assuming this has been thrown because delete file/folder failed.
                t = e;
            }
            try {
                Thread.sleep(200);
            } catch (InterruptedException e) {
                // Ignoring interrupted exception.
            }
        }
        if (null != t)
            throw new PlatformException("Failed to remove database folder at " + dbLocation.getAbsolutePath(), t);
    }
    public void resetWorkspace(IProgressMonitor monitor, ArrayList<String> fileFilter) throws PlatformException, IllegalStateException, IOException {
        File file = Platform.getLocation().toFile();
        if (null != fileFilter)
            FileUtils.deleteAllWithFilter(file , fileFilter);
        resetDatabase(monitor);
    }

    private static Path tryGetInstallLocation() {
        Location l = Platform.getInstallLocation();
        return l == null ? null : new File(l.getURL().getPath()).toPath();
    }

    public Path databaseExists() {
        Path dbLocation = dbLocation();
        if(Files.exists(dbLocation))
            return dbLocation;
        return null;
    }
    
    public Path dbLocation() {
        Path workspaceLocation = Platform.getLocation().toFile().toPath();
        return workspaceLocation.resolve("db");
    }

    /**
     * Start-up the platform. The procedure consists of 8 steps. Once everything
     * is up and running, all fields are set property.
     * <p>
     *
     * If workspacePolicy is FixErrors, there is an attempt to fix unexpected
     * errors. It includes installing database files, installing ontologies, and
     * installing project features.
     * <p>
     *
     * In Simantics Workbench this is handled in
     * <code>SimanticsWorkbenchAdvisor#openWindows()</code>.
     * <p>
     *
     * If remote server is given, simantics plaform takes connection there
     * instead of local server at "db/".
     *
     * @param workspacePolicy action to take on workspace/database related
     *        errors
     * @param ontologyPolicy action to take on ontology mismatch
     * @param progressMonitor optional progress monitor
     * @param userAgent interface for resorting to user feedback during platform
     *        startup or <code>null</code> to resort to default measures
     * @throws PlatformException
     */
    public synchronized SessionContext startUp(String databaseDriverId, IProgressMonitor progressMonitor, RecoveryPolicy workspacePolicy,
            OntologyRecoveryPolicy ontologyPolicy, boolean requireSynchronize, PlatformUserAgent userAgent)
    throws PlatformException
    {

        assert(!running);
        LOGGER.info("Beginning of SimanticsPlatform.startUp");

        SubMonitor monitor = SubMonitor.convert(progressMonitor, 1000);

        // For debugging on what kind of platform automatic tests are running in
        // case there are problems.
        if ("true".equals(System.getProperty("org.simantics.dumpBundleState")))
            dumpPlatformBundleState();

        // 0. Consult all startup extensions before doing anything with the workspace.
        StartupExtensions.consultStartupExtensions();
        LOGGER.info("Consulted platform pre-startup extensions");

        // 0.1. Clear all temporary files
        Simantics.clearTemporaryDirectory();
        LOGGER.info("Cleared temporary directory");

        // 0.2 Clear VariableRepository.repository static map which holds references to SessionImplDb
        VariableRepository.clear();

        // 0.2.1 Activate org.simantics.scl.osgi to prime the SCL compiler early.
        @SuppressWarnings("unused")
        ModuleRepository modRepo = SCLOsgi.MODULE_REPOSITORY;
        File location = new File(Platform.getInstanceLocation().getURL().getFile());
        modRepo.setLocation(location.toPath());

        // 0.3 Handle baseline database before opening db
        @SuppressWarnings("unused")
        boolean usingBaseline = DatabaseBaselines.handleBaselineDatabase(tryGetInstallLocation(), databaseExists() != null);

        // 1. Assert there is a database at <workspace>/db
        SessionDescriptor sessionDescriptor = setupDatabase(databaseDriverId, monitor.newChild(200, SubMonitor.SUPPRESS_NONE), workspacePolicy, userAgent);
        session = sessionDescriptor.getSession();
        LOGGER.info("Database setup complete");

        // 1.1. Setup SCL module repository with DB support
        SCLOsgi.MODULE_REPOSITORY.setAdvisor(new ModuleCompilationOptionsAdvisor() {
            ModuleCompilationOptions options = null;
            @Override
            public ModuleCompilationOptions getOptions(String moduleName) {
                if(options == null) {
                    options = new ModuleCompilationOptions(false);
                    options.collectDebugInfo = Activator.getContext().getProperty("osgi.dev") != null; //$NON-NLS-1$
                    options.classLoaderProvider = new StandardStoredModuleSupport(session);
                }
                return options; 
            }
        });

        
        // 1.2. Let's try to compile StandardLibrary asynchronously to speed up
        // the compilation when we actually need it the first time
        LOGGER.info("Warming up SCL-compiler with StandardLibrary");
        ForkJoinPool.commonPool().submit(() -> {
            Failable<org.simantics.scl.compiler.module.Module> module = SCLOsgi.MODULE_REPOSITORY.getModule("StandardLibrary");
            LOGGER.info("StandardLibrary compiled {}", module);
        });
        
        // 2. Delete all indexes if we cannot be certain they are up-to-date
        //    A full index rebuild will be done later, before project activation.
        XSupport support = session.getService(XSupport.class);
        if (support.rolledback()) {
            try {
                DatabaseIndexing.deleteAllIndexes();
            } catch (IOException e) {
                throw new PlatformException(e);
            }
        }

        // 3. Assert all graphs, and correct versions, are installed to the database
        synchronizeOntologies(monitor.newChild(400, SubMonitor.SUPPRESS_NONE), ontologyPolicy, requireSynchronize);

        // 4. Assert simantics.cfg exists
        boolean installProject = assertConfiguration(monitor.newChild(25, SubMonitor.SUPPRESS_NONE),workspacePolicy);

        // 5. Assert Project Resource is installed in the database
        installProject = assertProject(monitor.newChild(25, SubMonitor.SUPPRESS_NONE), workspacePolicy, installProject);

        // 6. Install all features into project, if in debug mode
        updateInstalledGroups(monitor.newChild(25), true); //installProject);
        LOGGER.info("Installed all features into project");

        // 7. Assert L0.Session in database for this session
        assertSessionModel(monitor.newChild(25, SubMonitor.SUPPRESS_NONE));

        session.getService(XSupport.class).setServiceMode(false, false);

        /*try {
            String message = "Flush query cache";
            monitor.setTaskName(message);
            LOGGER.info(message);
            session.syncRequest((Write) graph -> {
                QueryControl qc = graph.getService(QueryControl.class);
                qc.flush(graph);
            });
        } catch (DatabaseException e) {
            LOGGER.error("Flushing queries failed.", e);
        }*/
        boolean loadProject = true;
        try {
            String message = "Open database session";
            monitor.setTaskName(message);
            LOGGER.info(message);
        	sessionContext = SimanticsPlatform.INSTANCE.createSessionContext(true);
        	// This must be before setSessionContext since some listeners might query this
            sessionContext.setHint(SimanticsKeys.KEY_PROJECT, SimanticsPlatform.INSTANCE.projectResource);

            Simantics.setSessionContext(sessionContext);

            // 1. Put ResourceBinding that throws an exception to General Bindings
            message = "Put ResourceBinding that throws an exception to General Bindings";
            LOGGER.info(message);
            simanticsBindings = new SimanticsBindings();
            Bindings.classBindingFactory.addFactory( simanticsBindings );

            Session session = sessionContext.getSession();
            session.registerService(Databoard.class, Bindings.databoard);

            // Register datatype bindings
            message = "Register datatype bindings";
            LOGGER.info(message);
            Bindings.defaultBindingFactory.getRepository().put(RGB.Integer.BINDING.type(), RGB.Integer.BINDING);
            Bindings.defaultBindingFactory.getRepository().put(Font.BINDING.type(), Font.BINDING);
            Bindings.defaultBindingFactory.getRepository().put(Datatypes.APPENDABLE_STRING, Bindings.APPENDABLE_STRING);

            if (support.rolledback() || sessionDescriptor.isFreshDatabase()) {
                message = "Rebuilding all indexes";
                LOGGER.info(message);
                monitor.setTaskName(message);
                try {
                    session.getService(IndexedRelations.class).fullRebuild(monitor.newChild(100), session);
                } catch (IndexException e) {
                    LOGGER.error("Failed to re-build all indexes", e);
                }
            } else {
                monitor.worked(100);
            }

            if(loadProject) {
                message = "Load project";
                monitor.setTaskName(message);
                LOGGER.info(message);
                project = Projects.loadProject(sessionContext.getSession(), SimanticsPlatform.INSTANCE.projectResource);
                sessionContext.setHint(ProjectKeys.KEY_PROJECT, project);
                monitor.worked(100);
                message = "Loading projects complete";
                LOGGER.info(message);

                message = "Activate project";
                monitor.setTaskName(message);
                LOGGER.info(message);
                project.activate();
                monitor.worked(100);
                LOGGER.info("Project activated");
            }

        } catch (DatabaseException e) {
            LOGGER.error("Platform startup failed.", e);
            throw new PlatformException(e);
        } catch (ProjectException e) {
            boolean hasStackTrace = e.getStackTrace().length > 0;
            if (!hasStackTrace)
                throw new PlatformException(e.getMessage(), hasStackTrace);
            throw new PlatformException(e, hasStackTrace);
        }

        running = true;

        // #7650: improve shutdown robustness in all applications that use the platform
        Runtime.getRuntime().addShutdownHook(shutdownHook);

        // Discard database session undo history at this point to prevent
        // the user from undoing any initialization operations performed
        // by the platform startup.
        SimanticsPlatform.INSTANCE.discardSessionUndoHistory();
        LOGGER.info("Discarded session undo history");

        return sessionContext;

    }

    public void registerServices(SessionContext context) {
        new GlobalServiceInitializer().initialize(session);
        session.registerService(MergingGraphRequestProcessor.class, new MergingGraphRequestProcessor("SessionService", session, 20));
        session.registerService(MergingDelayedWriteProcessor.class, new MergingDelayedWriteProcessor(session, 20));
    }


    public SessionContext createSessionContext(boolean init) throws PlatformException {
        try {
            // Construct and initialize SessionContext from Session.
            SessionContext sessionContext = SessionContext.create(session, init);
            String message = "Session context created";
            LOGGER.info(message);
            if (init) {
                registerServices(sessionContext);
                message = "Session services registered";
                LOGGER.info(message);
            }
            return sessionContext;
        } catch (DatabaseException e) {
            throw new PlatformException(e);
        }
    }

    /**
     * Perform normal shutdown for the Simantics Platform.
     *
     * @param progressMonitor optional progress monitor
     * @throws PlatformException
     * @see {@link #shutdown(IProgressMonitor, boolean)}
     */
    public synchronized void shutdown(IProgressMonitor progressMonitor) throws PlatformException {
        shutdown(progressMonitor, true);
    }

    /**
     * Shutdown Simantics Platform.
     *
     * In Simantics Workbench this is handled in
     * <code>SimanticsWorkbenchAdvisor#disconnectFromWorkspace</code>.
     *
     * @param progressMonitor
     *            optional progress monitor
     * @param clearTemporaryFiles
     *            allow or prevent deletion of temporary files at the end of the
     *            shutdown procedure
     * @throws PlatformException
     */
    public synchronized void shutdown(IProgressMonitor progressMonitor, boolean clearTemporaryFiles) throws PlatformException
    {
        SubMonitor progress = SubMonitor.convert(progressMonitor, 100);
        PlatformException platformException = null;
        try {
            progress.subTask("Close Project");
            if (project != null) {
                project.safeDispose();
            }
            progress.worked(10);

            // NOP at the moment
            TimedSessionCache.close();

            progress.subTask("Thread pools");
            ThreadUtils.shutdown();
            ExecutorWorker.shutdown();
            progress.worked(5);

            running = false;
            progress.subTask("Close Database Session");
            if (sessionContext != null) {
                Session s = sessionContext.peekSession();
                if (s != null) {
                    progress.subTask("Flushing Index Caches");
                    try {
                        Simantics.flushIndexCaches(progress.newChild(20), s);
                    } catch (Throwable t) {
                        LOGGER.error("Failed to flush index caches.", t);
                    }

                    if("true".equals(System.getProperty("org.simantics.db.persistQueries"))) {
                        progress.subTask("Saving Queries");
                        Simantics.saveQueries(s);
                    }
                    
                    if("true".equals(System.getProperty("org.simantics.db.purgeOnExit"))) {
                        progress.subTask("Emptying Trash Bin and Purging DB");
                        try {
                            Layer0Utils.emptyTrashBin();
                        } catch (ServiceException e) {
                            throw new ProjectException(e);
                        }
                    }
                    
                }

                progress.subTask("Close Database Session");
                sessionContext.safeDispose();
                sessionContext = null;
                Simantics.setSessionContext(null);
            }
            if (simanticsBindings != null) {
                Bindings.classBindingFactory.removeFactory( simanticsBindings );
                simanticsBindings = null;
            }

            // Make sure Simantics clipboard doesn't store unwanted session data references.
            Simantics.setClipboard(new SimanticsClipboardImpl());

            progress.worked(50);

            session = null;
            projectResource = null;
            currentDatabaseDriver = null;

            DependenciesRelation.assertFinishedTracking();

        } catch (Exception e) {
            platformException = new PlatformException("Failed to shutdown Simantics Platform", e);
        }

        progress.worked(10);
        progress.subTask("Shutting down database");
        try {
            if (null != databasebManagement)
                databasebManagement.shutdown();
        } catch (Throwable t) {
            LOGGER.error("Database shutdown failed.", t);
        }
        progress.worked(10);

        progress.subTask("Clear index status");
        try {
            // Everything ok, clear index dirty state.
            DatabaseIndexing.clearAllDirty();
        } catch (IOException e) {
            LOGGER.error("Problems encountered while refreshing database index states, see exception for details.", e);
        }
        progress.worked(5);

        if (clearTemporaryFiles) {
            progress.subTask("Clearing Workspace Temporary Directory");
            try {
                Simantics.clearTemporaryDirectory();
            } catch (Throwable t) {
                LOGGER.error("Failed to clear the temporary directory.", t);
            }
        }
        progress.worked(10);
        if (null != platformException)
            throw platformException;

        // #7650: improve shutdown robustness in all applications that use the platform
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
    }

    // TODO: consider removing this in the future ??
    @Override
    public void stateChanged(LifecycleState newState) {
        if(newState == LifecycleState.CLOSED) {
            if(running) {
                if(Platform.isRunning()) {
                    mainThread.interrupt();
                }
            }
        }
    }

    /**
     * @return <code>true</code> if discard was successful, <code>false</code>
     *         if there was no session, {@link UndoRedoSupport} or
     *         {@link UndoContext} to discard through
     */
    public boolean discardSessionUndoHistory() {
        Session s = session;
        if (s != null) {
            UndoRedoSupport urs = s.peekService(UndoRedoSupport.class);
            if (urs != null) {
                UndoContext uc = urs.getUndoContext(s);
                if (uc != null) {
                    uc.clear();
                    return true;
                }
            }
        }
        return false;
    }

    public void reconnect(String databaseDriverId) throws Exception {
        // Starts database server.
        if (currentDatabaseDriver != null)
            databaseDriverId = currentDatabaseDriver;
        SimanticsPlatform.INSTANCE.startUp(databaseDriverId, null, RecoveryPolicy.ThrowError, OntologyRecoveryPolicy.ThrowError, true, null);
    }

    private void dumpPlatformBundleState() {
        BundleDescription[] bs = Platform.getPlatformAdmin().getState().getBundles();
        System.out.println("Total bundles: " + bs.length);
        for (BundleDescription b : bs) {
            System.out.format("%-80s @ %s\n", b.toString(), b.getLocation());
        }
    }

    private Ini loadOrCreateDatabaseIni(Path path, String databaseDriverId)
            throws InvalidFileFormatException, IOException
    {
        File f = path.toFile();
        Ini dbIni = Files.isRegularFile(path) ? new Ini(f) : new Ini();
        String iniId = dbIni != null ? dbIni.get("driver", "id") : null;
        if (iniId == null) {
            dbIni.put("driver", "id", databaseDriverId);
            dbIni.store(f);
        }
        return dbIni;
    }

}
