/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.internal;

import java.util.Hashtable;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.simantics.db.Disposable;
import org.simantics.db.management.ISessionContext;
import org.simantics.utils.datastructures.cache.SoftTimedCache;
import org.simantics.utils.datastructures.disposable.IDisposable;
/**
 * @author Tuukka Lehtonen
 */
public class TimedSessionCache extends SoftTimedCache</*ServerAddress*/Object, ISessionContext> {

    private static ServiceRegistration<?> service       = null;
    private static BundleContext          bundleContext = null;
    private static boolean                disposed      = false;

    TimedSessionCache() {
        super("Database Session Cache Timer");
    }

    /**
     * Invoked by the bundle activator to initialize the cache service.
     * 
     * @param context the bundle context to register the service with
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public synchronized static void initialize(BundleContext context) {
        bundleContext = context;
        service = context.registerService(TimedSessionCache.class.getName(), new TimedSessionCache(), new Hashtable());
        disposed = false;
    }

    /**
     * Invoked by the bundle activator to close the cache service.
     */
    public synchronized static void close() {
        if (service != null) {
            TimedSessionCache cache = getCache();
            cache.clear();
            
            service.unregister();
            service = null;
            disposed = true;
        }
        bundleContext = null;
    }

    public static synchronized TimedSessionCache getCache() {
        if (disposed)
            throw new IllegalStateException("cache service is disposed");
        if (service == null)
            throw new IllegalStateException("cache service not initialized");
        return (TimedSessionCache) bundleContext.getService(service.getReference());
    }

    @Override
    protected void disposeValue(ISessionContext v) {
        super.disposeValue(v);
        // Implementing Disposable allows immediate disposal of the value
        // object instead of leaving the disposal up to garbage
        // collection and finalization.
        if (v instanceof IDisposable) {
            ((IDisposable) v).safeDispose();
        } else if (v instanceof Disposable) {
            ((Disposable) v).dispose();
        }
    }

}
