/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.WriteBindings;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.service.ClusteringSupport;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.graph.db.AbstractImportAdvisor2;
import org.simantics.graph.db.TransferableGraphImporter;
import org.simantics.graph.representation.Extensions;
import org.simantics.graph.representation.Root;
import org.simantics.layer0.Layer0;
import org.simantics.project.management.DatabaseManagement;
import org.simantics.project.management.GraphBundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OntologyImportAdvisor extends AbstractImportAdvisor2 {

	private static final Logger LOGGER = LoggerFactory.getLogger(OntologyImportAdvisor.class);

	final private GraphBundle tg;
	final private DatabaseManagement mgmt;
	final private String rootName;
	private Resource ontology;
	private Resource ontologyName;

	public OntologyImportAdvisor(GraphBundle tg, DatabaseManagement mgmt) {
		this.tg = tg;
		this.mgmt = mgmt;
		if(tg.getName().contains("@")) {
			rootName = tg.getName().substring(tg.getName().lastIndexOf("/")+1);
		} else {
			rootName = tg.getName().substring(tg.getName().lastIndexOf("/")+1) + "-" + tg.getMajor() + "." + tg.getMinor();
		}
	}

	@Override
	public Resource getTarget() {
		return null;
	}

	@Override
	public Resource analyzeRoot(ReadGraph graph, Root root) throws DatabaseException {
		return null;
	}

	@Override
	public Resource createRoot(WriteOnlyGraph graph, Root root) throws DatabaseException {
		return createRoot(graph, root, null); 
	}

	@Override
	public Resource createRoot(WriteOnlyGraph graph, Root root, Resource resource) throws DatabaseException {
		if(resource == null) resource = graph.newResource();
		addRootInfo(root, root.name, resource);
		return resource;
	}

	int[] getClustering(TransferableGraphImporter process) {
		var extensions = process.getExtensions();
		Variant v = extensions.get(Extensions.CLUSTERING);
		if(v == null) return null;
		try {
			return (int[])v.getValue(Bindings.INT_ARRAY);
		} catch (AdaptException e) {
			LOGGER.error("Error while reading clustering information", e);
			return null;
		}
	}

	@Override
	public void beforeWrite(WriteOnlyGraph graph, TransferableGraphImporter process) throws DatabaseException {
		int[] clustering = getClustering(process);
		if(clustering == null) {
			// If there is no clustering available make sure that the ontology itself is a cluster set
			ontology = graph.newResource(graph.getDefaultImmutableClusterSet());
			ontologyName = graph.newResource(graph.getDefaultImmutableClusterSet());
			graph.newClusterSet(ontology);
			graph.setClusterSet4NewResource(ontology);
		}
	}

	@Override
	public void afterWrite(WriteOnlyGraph graph, TransferableGraphImporter process) throws DatabaseException {

		try {
			long[] ids = process.getResourceIds(graph.getService(SerialisationSupport.class));
			tg.setResourceArray(ids);
			mgmt.createGraphBundle(graph, tg);
		} catch (DatabaseException e) {
			throw new RuntimeDatabaseException(e);
		}

	}

	@Override
	public boolean allowImmutableModifications() {
		return true;
	}

	@Override
	public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent,
			String name) throws DatabaseException {

		if(ontology != null && rootName.equals(name)) {

			// In this case there is no clustering info available
			Layer0 L0 = graph.getService(Layer0.class);
			graph.claim(ontologyName, L0.InstanceOf, null, L0.String);
			graph.claimValue(ontologyName, name, WriteBindings.STRING);
			graph.claim(ontology, L0.HasName, L0.NameOf, ontologyName);
			return ontology;

		} else {

			return process.createChild(graph, parent, null, name);

		}
	}

	@Override
	public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent, Resource child,
			String name) throws DatabaseException {

		if(ontology != null && rootName.equals(name)) {

			// In this case there is no clustering info available
			Layer0 L0 = graph.getService(Layer0.class);
			graph.claim(ontologyName, L0.InstanceOf, null, L0.String);
			graph.claimValue(ontologyName, name, WriteBindings.STRING);
			graph.claim(ontology, L0.HasName, L0.NameOf, ontologyName);
			graph.claim(ontology, L0.PartOf, L0.ConsistsOf, parent);
			return ontology;

		} else {

			if(rootName.equals(name)) {

				{
					// Legacy special case - the tg specifies clustering but does not declare ontology as cluster set
					ClusteringSupport cs = graph.getService(ClusteringSupport.class);
					if(!cs.isClusterSet(child))
						graph.newClusterSet(child);
				}

				// We have clustering, arrange ontology name to the root cluster set for browsing
				Layer0 L0 = graph.getService(Layer0.class);
				Resource ontologyName = graph.newResource(graph.getRootLibrary());
				graph.claim(ontologyName, L0.InstanceOf, null, L0.String);
				graph.claimValue(ontologyName, name, WriteBindings.STRING);
				graph.claim(child, L0.HasName, L0.NameOf, ontologyName);
				graph.claim(child, L0.PartOf, L0.ConsistsOf, parent);
				return child;

			} else {

				return process.createChild(graph, parent, child, name);

			}

		}
	}

}
