package org.simantics;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.common.utils.Versions;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.layer0.Layer0;
import org.simantics.platform.ui.PlatformUIResource;


/**
 * @author Antti Villberg
 */
public class NameLabelUtil {

	public static NameLabelMode getNameLabelMode(RequestProcessor processor) {
		try {
			Resource project = Simantics.getProjectResource(); 
			return processor.syncRequest(new ResourceRead<NameLabelMode>(project) {
				@Override
				public NameLabelMode perform(ReadGraph graph) throws DatabaseException {
					PlatformUIResource UI = PlatformUIResource.getInstance(graph);
					String value = graph.getPossibleRelatedValue(resource, UI.hasNameLabelMode, Bindings.STRING);
					return NameLabelMode.fromString(value);
				}
			});
		} catch (DatabaseException e) {

			return NameLabelMode.getDefault();
		}
	}	

	public static void setNameLabelMode(RequestProcessor processor, final NameLabelMode mode) {
		try {
			VirtualGraphSupport support = processor.getService(VirtualGraphSupport.class);
			VirtualGraph vg = support.getWorkspacePersistent("preferences");
			processor.syncRequest(new WriteRequest(vg) {
				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					PlatformUIResource UI = PlatformUIResource.getInstance(graph);
					Resource project = Simantics.getProjectResource(); 
					graph.deny(project, UI.hasNameLabelMode);
					graph.claim(project, UI.hasNameLabelMode, mode.asResource(UI));
				}
			});
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
	}	

	public static String modalName(ReadGraph graph, Resource resource) throws DatabaseException {
		NameLabelMode mode = NameLabelUtil.getNameLabelMode(graph);
		return modalName(graph, resource, mode);
	}

	public static String modalName(ReadGraph graph, Variable variable) throws DatabaseException {
		NameLabelMode mode = NameLabelUtil.getNameLabelMode(graph);
		return modalName(graph, variable, mode);
	}

	public static String modalName(ReadGraph graph, Resource resource, NameLabelMode mode) throws DatabaseException {
		switch (mode) {
		case NAME: {
			return Versions.getStandardNameString(graph, resource);
		}
		case LABEL: {
			Layer0 L0 = Layer0.getInstance(graph);
			String label = graph.getPossibleRelatedValue2(resource, L0.HasLabel, Bindings.STRING);
			if(label == null || label.isEmpty()) label = "no label (" + Versions.getStandardNameString(graph, resource) + ")";
			return label;
		}
		case NAME_AND_LABEL: {
			Layer0 L0 = Layer0.getInstance(graph);
			String name = Versions.getStandardNameString(graph, resource);
			String l = graph.getPossibleRelatedValue2(resource, L0.HasLabel, Bindings.STRING);
			String label = name + ((l != null && !l.isEmpty()) ? " (" + l + ")" : "");
			return label;
		}
		case LABEL_AND_NAME: {
			Layer0 L0 = Layer0.getInstance(graph);
			String name = Versions.getStandardNameString(graph, resource);
			String l = graph.getPossibleRelatedValue2(resource, L0.HasLabel, Bindings.STRING);
			String label = ((l != null && !l.isEmpty()) ? l : "no label") + " (" + name + ")";
			return label;
		}
		default:
			throw new UnsupportedOperationException("unsupported mode: " + mode);
		}
	}

	public static String modalName(ReadGraph graph, Variable variable, NameLabelMode mode) throws DatabaseException {
		switch (mode) {
		case NAME: {
			return name(graph, variable);
		}
		case LABEL: {
			String label = label(graph, variable);
			if(label == null || label.isEmpty()) label = "no label (" + name(graph, variable) + ")";
			return label;
		}
		case NAME_AND_LABEL: {
			String name = name(graph, variable);
			String l = label(graph, variable);
			String label = name + ((l != null && !l.isEmpty()) ? " (" + l + ")" : "");
			return label;
		}
		case LABEL_AND_NAME: {
			String name = name(graph, variable);
			String l = label(graph, variable);
			String label = ((l != null && !l.isEmpty()) ? l : "no label") + " (" + name + ")";
			return label;
		}
		default:
			throw new UnsupportedOperationException("unsupported mode: " + mode);
		}
	}

	public static String modalName(String name, String label, NameLabelMode mode) {
		switch (mode) {
		case NAME:
			return name;
		case LABEL:
			return label == null || label.isEmpty() ? label = "no label (" + name + ")" : label;
		case NAME_AND_LABEL:
			return name + ((label != null && !label.isEmpty()) ? " (" + label + ")" : "");
		case LABEL_AND_NAME:
			return ((label != null && !label.isEmpty()) ? label : "no label") + " (" + name + ")";
		default:
			throw new UnsupportedOperationException("unsupported mode: " + mode);
		}
	}

	private static String name(ReadGraph graph, Variable v) throws DatabaseException {
		Resource r = v.getPossibleRepresents(graph);
		return r != null ? Versions.getStandardNameString(graph, r) : v.getName(graph);
	}

	private static String label(ReadGraph graph, Variable v) throws DatabaseException {
		return v.getPossiblePropertyValue(graph, Variables.LABEL, Bindings.STRING);
	}

}
