package org.simantics;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IProduct;
import org.eclipse.core.runtime.Platform;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Jani Simomaa
 *
 */
public final class BuildInfo {

    private static final Logger LOGGER = LoggerFactory.getLogger(BuildInfo.class);
    private static final Properties PROPS;
    private static final String ALL;

    public static final String DEFAULT_BUILDINFO_PROPERTIES_FILE = "buildinfo.properties";
    public static final String BUILDINFO_PROPERTIES_PROP = "buildinfo.properties.location";

    public static final String BUILD_INFO_PROPERTIES_FILE = System.getProperty(BUILDINFO_PROPERTIES_PROP,
            DEFAULT_BUILDINFO_PROPERTIES_FILE);

    static {
        if (LOGGER.isDebugEnabled())
            LOGGER.debug("Trying to load build information from {}", BUILD_INFO_PROPERTIES_FILE);
        Properties buildinfo = new Properties();
        Path propertiesFile = Paths.get(BUILD_INFO_PROPERTIES_FILE);
        if (Files.exists(propertiesFile)) {
            try {
                buildinfo.load(Files.newInputStream(propertiesFile));
                if (LOGGER.isDebugEnabled())
                    LOGGER.debug(buildinfo.toString());
            } catch (IOException e) {
                LOGGER.error("Could not load build information from {}", propertiesFile.toAbsolutePath(), e);
            }
        } else {
            if (LOGGER.isDebugEnabled())
                LOGGER.debug("Build information not loaded as {} does not exist", propertiesFile.toAbsolutePath());
        }
        // add information from .product if possible
        collectInformationFromActiveProduct(buildinfo);
        
        PROPS = buildinfo;
        StringBuilder sb = new StringBuilder();
        TreeMap<String, String> orderedProps = new TreeMap<>((o1, o2) -> o1.compareTo(o2));
        for (Entry<Object, Object> entry : PROPS.entrySet())
            orderedProps.put(entry.getKey().toString(), entry.getValue().toString());
        for (Entry<String, String> entry : orderedProps.entrySet()) {
            sb.append(entry.getKey()).append("=").append(entry.getValue()).append("\n");
        }
        ALL = sb.toString();
    }

    private static void collectInformationFromActiveProduct(Properties buildinfo) {
        IProduct product = Platform.getProduct();
        if (product != null) {
            String productName = product.getName() != null ? product.getName() : "<not-set>";
            String productApplication = product.getApplication() != null ? product.getApplication() : "<not-set>";
            String productDescription = product.getDescription() != null ? product.getDescription() : "<not-set>";
            String productId = product.getId() != null ? product.getId() : "<not-set>";
                
            buildinfo.setProperty("product.name", productName);
            buildinfo.setProperty("product.application", productApplication);
            buildinfo.setProperty("product.description", productDescription);
            buildinfo.setProperty("product.id", productId);
        }
    }

    /**
     * @param key
     * @return
     */
    public static String get(String key) {
        return PROPS.getProperty(key, "<not-set>");
    }

    /**
     * @return
     */
    public static String all() {
        return ALL;
    }

    /**
     * @return
     */
    public static Map<String, String> asMap() {
        return PROPS.entrySet().stream().collect(Collectors.toMap(k -> String.valueOf(k), v -> String.valueOf(v)));
    }
}
