/*******************************************************************************
 * Copyright (c) 2019, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.scl.db;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.cojen.classfile.TypeDesc;
import org.simantics.Simantics;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.GraphHints;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.SyncListenerAdapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UnaryAsyncRead;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.request.Read;
import org.simantics.db.service.ClusterControl;
import org.simantics.db.service.QueryControl;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.db.service.XSupport;
import org.simantics.layer0.utils.triggers.IActivationManager;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.environment.specification.EnvironmentSpecification;
import org.simantics.scl.compiler.errors.DoesNotExist;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.internal.codegen.types.JavaTypeTranslator;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ImportFailureException;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.runtime.RuntimeModule;
import org.simantics.scl.compiler.top.ValueNotFound;
import org.simantics.scl.compiler.types.TCon;
import org.simantics.scl.compiler.types.TFun;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.exceptions.MatchException;
import org.simantics.scl.compiler.types.util.MultiFunction;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.reflection.ValueNotFoundException;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.reporting.SCLReporting;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.scl.runtime.tuple.Tuple;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings({"rawtypes", "unchecked"})
public class SCLFunctions {

    private static final Logger LOGGER = LoggerFactory.getLogger(SCLFunctions.class);

    public static final String GRAPH = "graph";

    public static <T> T safeExec(final Function f) {
        try {
            return (T)f.apply(Tuple0.INSTANCE);
        } catch (Throwable t) {
            LOGGER.error("safeExec caught exception", t);
            return null;
        }
    }

    public static Function resolveFunction(RuntimeModule rm, String function) throws ValueNotFound {
        return (Function)rm.getValue(function);
    }
    
    private static SCLValue resolveSCLValue(RuntimeModule rm, String function) throws ValueNotFound {
        return rm.getModule().getValue(function);
    }
    
    private static RuntimeModule resolveRuntimeModule(String module) throws ValueNotFound {
        Failable<RuntimeModule> f = SCLOsgi.MODULE_REPOSITORY.getRuntimeModule(module);
        if(f.didSucceed())
            return f.getResult();
        else if(f == DoesNotExist.INSTANCE)
            throw new ValueNotFound("Didn't find module " + module);
        else
            throw new ValueNotFound(((Failure)f).toString());
    }
    
    private static int getArity(TFun fun) {
      Type range = fun.range;
      if(range instanceof TFun) {
          return 1 + getArity((TFun)range);
      } else {
          return 1;
      }
    }
    
    private static List<TCon> getEffects(SCLValue value) throws ValueNotFoundException, ValueNotFound, MatchException {
        Type type = value.getType();
        if(!(type instanceof TFun))
            return Collections.emptyList();
        int argn = getArity((TFun)type);
        return getEffects(value, argn);
    }

    private static List<TCon> getEffects(SCLValue value, int argn) throws ValueNotFoundException, ValueNotFound, MatchException {
        Type type = value.getType();
        if(!(type instanceof TFun))
            return Collections.emptyList();
        MultiFunction mfun = Types.matchFunction(type, argn);
        ArrayList<TCon> concreteEffects = new ArrayList<>();
        mfun.effect.collectConcreteEffects(concreteEffects);
        return concreteEffects;
    }

    public static List<TCon> getEffects(RuntimeModule rm, String function) throws ValueNotFoundException, ValueNotFound, MatchException {
        return getEffects(resolveSCLValue(rm, function));
    }

    public static List<TCon> getEffects(String module, String function) throws ValueNotFoundException, ValueNotFound, MatchException {
        return getEffects(resolveSCLValue(resolveRuntimeModule(module), function));
    }
    
    private static <T> T evaluate(Function function, Object ... args) {
        return (T)function.applyArray(args);
    }

    private static <T> T evaluate(RuntimeModule rm, String function, Object ... args) throws ValueNotFound {
        return evaluate(resolveFunction(rm, function), args);
    }

    public static <T> T evaluate(String module, String function, Object ... args) throws ValueNotFound {
        return evaluate(resolveRuntimeModule(module), function, args);
    }

    public static <T> T evaluateDB(String module, String function, Object ... args) throws DatabaseException {
        try {
            RuntimeModule rm = resolveRuntimeModule(module);
            List<TCon> effects = getEffects(resolveSCLValue(rm, function), args.length);
            Function f = resolveFunction(rm, function);
            if(effects.contains(Types.WRITE_GRAPH)) {
                return syncWrite(f, args);
            } else if(effects.contains(Types.READ_GRAPH)) {
                return syncRead(f, args);
            } else {
                return evaluate(f, args);
            }
        } catch (ValueNotFound e) {
            throw new DatabaseException("SCL Value not found: " + e.name);
        } catch (Throwable t) {
            if (t instanceof DatabaseException)
                throw (DatabaseException) t;
            throw new DatabaseException(t);
        }
    }
    
    public static <T> T evaluateGraph(String module, String function, ReadGraph graph, Object ... args) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        SCLContext.push(context);
        Object oldGraph = context.put(GRAPH, graph);
        try {
            return evaluateDB(module, function, args);
        } finally {
            context.put(GRAPH, oldGraph);
            SCLContext.pop();
        }
    }

    public static void runWithGraph(ReadGraph graph, Runnable r) {
        final SCLContext context = SCLContext.getCurrent();
        SCLContext.push(context);
        Object oldGraph = context.put(GRAPH, graph);
        try {
            r.run();
        } finally {
            context.put(GRAPH, oldGraph);
            SCLContext.pop();
        }
    }

    private static Object[] NO_ARGS = new Object[] { Tuple0.INSTANCE };

    public static <T> void asyncRead(final Function f) throws DatabaseException {
        asyncRead(f, NO_ARGS);
    }

    public static void asyncRead(final Function f, final Object ... args) throws DatabaseException {
        final SCLContext context = SCLContext.createDerivedContext();
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                SCLContext.push(context);
                context.put(GRAPH, graph);
                try {
                    f.applyArray(args);
                } finally {
                    SCLContext.pop();
                }
            }
        });
    }
    
    public static <T> T syncRead(final Function f) throws DatabaseException {
        return syncRead(f, NO_ARGS);
    }
    
    public static <T> T syncRead(final Function f, final Object ... args) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        Object graph = context.get(GRAPH);
        if (graph != null) {
            return (T)f.applyArray(args);
        } else {
            return Simantics.getSession().syncRequest(new Read<T>() {
                @Override
                public T perform(ReadGraph graph) throws DatabaseException {
                    SCLContext.push(context);
                    ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                    try {
                        return (T)f.applyArray(args);
                    } finally {
                        context.put(GRAPH, oldGraph);
                        SCLContext.pop();
                    }
                }
            });
        }
    }

    public static void asyncWrite(final Function f) throws DatabaseException {
        asyncWrite(f, NO_ARGS);
    }

    public static void asyncWrite(final Function f, final Object ... args) throws DatabaseException {
        if (Simantics.peekSession() != null) {
            Simantics.getSession().asyncRequest(new WriteRequestEvents(f, args));
        } else {
            LOGGER.warn("No session available for asynchronous write requests");
        }
    }
    
    public static <T> T syncWrite(final Function f) throws DatabaseException {
        return syncWrite(f, NO_ARGS);
    }

    public static <T> T syncWrite(final Function f, final Object ... args) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        Object graph = context.get(GRAPH);
        if (graph != null && graph instanceof WriteGraph) {
            return (T)f.applyArray(args);
        } else {
            if (graph != null) {
                LOGGER.error(
                        "SCLContext {} for current thread {} contains an existing graph object but it is not WriteGraph - Somewhere is a function that forgets to remove the graph from the context!!",
                        context, Thread.currentThread());
            }
            return Simantics.getSession().syncRequest(new WriteResultRequestEvents<T>(f, args));
        }
    }
    
    public static <T> T delayedSyncWrite(final Function f) throws DatabaseException {

        DelayedWriteRequestEvents<T> request = new DelayedWriteRequestEvents(f);
    
        final SCLContext context = SCLContext.getCurrent();
        Object graph = context.get(GRAPH);
        if (graph != null) {
            if (graph instanceof WriteGraph) {
                ((WriteGraph)graph).syncRequest(request);
            } else {
                throw new DatabaseException("Caller is inside a read transaction.");
            }
        } else {
            Simantics.getSession().syncRequest(request);
        }
        
        return request.dc.get();
        
    }

    public static <T> T virtualSyncWriteMem(WriteGraph graph, String virtualGraphId, final Function f) throws DatabaseException {
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getMemoryPersistent(virtualGraphId);
        return graph.syncRequest(new WriteResultRequestEvents<T>(vg, f, Tuple0.INSTANCE));
    }
    
    public static <T> T virtualSyncWriteWS(WriteGraph graph, String virtualGraphId, final Function f) throws DatabaseException {
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getWorkspacePersistent(virtualGraphId);
        return graph.syncRequest(new WriteResultRequestEvents<T>(vg, f, Tuple0.INSTANCE));
    }
    
    public static <T> T readValue(final String uri) throws DatabaseException {
        return Simantics.getSession().syncRequest(new Read<T>() {
            @Override
            public T perform(ReadGraph graph) throws DatabaseException {
                return Variables.getVariable(graph, uri).getValue(graph);
            }
        });
    }
    
    public static <T> void writeValue(final String uri, final T value) throws DatabaseException {
        Simantics.getSession().syncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                Variables.getVariable(graph, uri).setValue(graph, value);
            }
        });
    }
    
    public static void activateOnce(Resource r) {
        Simantics.getSession().getService(IActivationManager.class).activateOnce(r);
    }
    
    public static void syncActivateOnce(WriteGraph graph, Resource r) throws DatabaseException {
        graph.getService(IActivationManager.class).activateOnce(graph, r);
    }

    public static Resource resourceFromId(ReadGraph graph, long id) throws DatabaseException, IOException {
        SerialisationSupport ss = graph.getService(SerialisationSupport.class);
        return ss.getResource(id);
    }
    
    public static void disableDependencies(WriteGraph graph) {
        Layer0Utils.setDependenciesIndexingDisabled(graph, true);       
    }
    
    public static void enableDependencies(WriteGraph graph) {
        Layer0Utils.setDependenciesIndexingDisabled(graph, false);       
    }
    
    public static void collectClusters() {
        Simantics.getSession().getService(ClusterControl.class).collectClusters(Integer.MAX_VALUE);
    }
    
    public static class SCLUnaryRead extends BinaryRead<Function1<Object,Object>, Object, Object> {

        public SCLUnaryRead(Function1<Object, Object> parameter1, Object parameter2) {
             super(parameter1, parameter2);
        }

        @Override
        public Object perform(ReadGraph graph) throws DatabaseException {
            return Simantics.applySCLRead(graph, parameter, parameter2);
        }

    }
    
    public static Object unaryQuery(ReadGraph graph, Function1<Object,Object> fn, Object value) throws DatabaseException {
        return graph.syncRequest(new SCLUnaryRead(fn, value));
    }

    public static Object unaryQueryCached(ReadGraph graph, Function1<Object,Object> fn, Object value) throws DatabaseException {
        return graph.syncRequest(new SCLUnaryRead(fn, value), TransientCacheAsyncListener.<Object>instance());
    }
    

    private static class Subquery extends UnaryRead<Function, Object> {

        public Subquery(Function q) {
            super(q);
        }

        @Override
        public Object perform(ReadGraph graph) throws DatabaseException {
            return Simantics.applySCLRead(graph, parameter, Tuple0.INSTANCE);
        }

    }

    private static class SubqueryA extends UnaryAsyncRead<Function, Object> {

        public SubqueryA(Function q) {
            super(q);
        }

        @Override
        public void perform(AsyncReadGraph graph, AsyncProcedure<Object> procedure) {
            try {
                procedure.execute(graph, Simantics.applySCLRead(graph, parameter, Tuple0.INSTANCE));
            } catch (DatabaseException e) {
                procedure.exception(graph, e);
            }
        }

    }

    public static Object subquery(ReadGraph graph, Function q) throws DatabaseException {
        return graph.syncRequest(new Subquery(q));
    }

    public static Object subqueryA(ReadGraph graph, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryA(q));
    }

    public static void subqueryP(AsyncReadGraph graph, Function q, Function executeCallback, Function1<Throwable, Tuple> exceptionCallback) throws DatabaseException {
        graph.asyncRequest(new SubqueryA(q), new AsyncProcedure() {
            @Override
            public void execute(AsyncReadGraph graph, Object result) {
                try {
                    Simantics.applySCLRead(graph, executeCallback, result);
                } catch (DatabaseException e) {
                    LOGGER.error("Exception while executing callback: " + executeCallback + " with result " + result, e);
                }
            }
            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                try {
                    Simantics.applySCLRead(graph, exceptionCallback, throwable);
                } catch (DatabaseException e) {
                    LOGGER.error("Exception while handling exception from function {} with callback {}", q, exceptionCallback, throwable);
                    LOGGER.error("Exception handler raised exception", e);
                }
            }
        });
    }

    public static Variant subqueryCRVariant(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRVariant(r, q), TransientCacheAsyncListener.instance());
    }

    public static Resource subqueryCRR(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRR(r, q), TransientCacheAsyncListener.instance());
    }

    public static Resource subqueryCRMR(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRR(r, q), TransientCacheAsyncListener.instance());
    }

    public static String subqueryCRS(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRS(r, q), TransientCacheAsyncListener.instance());
    }

    public static String subqueryCRMS(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRMS(r, q), TransientCacheAsyncListener.instance());
    }

    public static List<Resource> subqueryCRResourceList(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRResourceList(r, q), TransientCacheAsyncListener.instance());
    }

    public static List<Statement> subqueryCRStatementList(ReadGraph graph, Resource r, Function q) throws DatabaseException {
        return graph.syncRequest(new SubqueryRStatementList(r, q), TransientCacheAsyncListener.instance());
    }

    public static Object subqueryC(ReadGraph graph, Function q) throws DatabaseException {
        return graph.syncRequest(new Subquery(q), TransientCacheAsyncListener.<Object>instance());
    }
    
    public static void subqueryL(ReadGraph graph, Function query, Function executeCallback, Function1<Throwable, Tuple> exceptionCallback, Function1<Tuple0, Boolean> isDisposedCallback) throws DatabaseException {
        graph.syncRequest(new Subquery(query), new SyncListenerAdapter<Object>() {
            @Override
            public void execute(ReadGraph graph, Object result) throws DatabaseException {
                Simantics.applySCLRead(graph, executeCallback, result);
            }
            
            @Override
            public void exception(ReadGraph graph, Throwable t) throws DatabaseException {
                Simantics.applySCLRead(graph, exceptionCallback, t);
            }
            
            @Override
            public boolean isDisposed() {
                return isDisposedCallback.apply(Tuple0.INSTANCE);
            }
        });
    }

    public static Object possibleFromDynamic(Type expectedType, String moduleName, Object value) {
    
        try {

        
            Failable<Module> failable = SCLOsgi.MODULE_REPOSITORY.getModule(moduleName);
            Module module = failable.getResult();
            
            RuntimeEnvironment env = SCLOsgi.MODULE_REPOSITORY.createRuntimeEnvironment(
            EnvironmentSpecification.of(moduleName, ""), module.getParentClassLoader());

            JavaTypeTranslator tr = new JavaTypeTranslator(env.getEnvironment());
            TypeDesc desc = tr.toTypeDesc(expectedType);
            String className = desc.getFullName();
            Class<?> clazz = env.getMutableClassLoader().loadClass(className);
            if (!clazz.isAssignableFrom(value.getClass()))
                return null;
            
        } catch (ImportFailureException | ClassNotFoundException e) {
        }
        return value;
    }

    public static void restrictQueries(ReadGraph graph, int amount, int step, int maxTimeInMs) {

        QueryControl qc = graph.getService(QueryControl.class);
        long start = System.currentTimeMillis();
        while(true) {
            int current = qc.count();
            if(current < amount) return;
            qc.gc(graph, step);
            long duration = System.currentTimeMillis() - start;
            if(duration > maxTimeInMs) return;
        }

    }

    public static int countQueries(ReadGraph graph) {

        QueryControl qc = graph.getService(QueryControl.class);
        return qc.count();

    }
    
    public static Object withGraphHintBoolean(ReadGraph graph, String key, boolean value, Function fn) {
        GraphHints old = graph.setHintValue(key, value);
        try {
            return fn.apply(Tuple0.INSTANCE);
        } finally {
            graph.setHints(old);
        }
    }
    
    public static Object withServiceMode(boolean allowImmutableWrites, boolean createAsImmutable, Function fn) {
        XSupport xs = Simantics.getSession().getService(XSupport.class);
        Pair<Boolean,Boolean> sm = xs.getServiceMode();
        try {
            xs.setServiceMode(allowImmutableWrites, createAsImmutable);
            return fn.apply(Tuple0.INSTANCE);
        } finally {
            xs.setServiceMode(sm.first, sm.second);
        }
    }

    public static void printStack() {
        try (var sw = new StringWriter()) {
            try (var pw = new PrintWriter(sw)) {
                new Exception().printStackTrace(pw);
            }
            SCLReportingHandler reportingHandler = SCLReporting.getCurrentReportingHandler();
            if(reportingHandler != null) {
                reportingHandler.print(sw.toString());
            } else {
                LOGGER.info(sw.toString());
            }
        } catch (IOException e) {
            // StringWriter and PrintWriter can't throw this.
            LOGGER.error("Closing stream failed", e);
        }
    }

}
