package org.simantics.scl.compiler.source.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.scl.compiler.common.names.Name;
import org.simantics.scl.compiler.elaboration.expressions.EExternalConstant;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.module.ConcreteModule;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.runtime.RuntimeModule;
import org.simantics.scl.compiler.source.ModuleSource;
import org.simantics.scl.compiler.source.PrecompiledModuleSource;
import org.simantics.scl.compiler.source.repository.procedural.ProceduralValueDefinition;
import org.simantics.scl.compiler.top.ValueNotFound;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ProceduralModuleSourceRepository implements ModuleSourceRepository {

    private static final Logger LOGGER = LoggerFactory.getLogger(ProceduralModuleSourceRepository.class);
    
    public static final String PREFIX = "procedural:";
    
    protected abstract ModuleRepository getModuleRepository(UpdateListener listener);
    
    @Override
    public Collection<String> getModuleNames() {
        return Collections.emptyList();
    }

    @Override
    public ModuleSource getModuleSource(String moduleName, UpdateListener listener) {
        if(!moduleName.startsWith(PREFIX))
            return null;
        
        String sourceModuleName = moduleName.substring(PREFIX.length());
        Failable<RuntimeModule> sourceModuleF = getModuleRepository(listener)
                .getRuntimeModule(sourceModuleName, listener);
        if(!sourceModuleF.didSucceed())
            return null;
        
        RuntimeModule sourceModule = sourceModuleF.getResult();
        Object values;
        try {
            values = sourceModule.getValue("values");
        } catch (ValueNotFound e) {
            return null; // normal
        } catch (Throwable e) {
            // abnormal exception
            LOGGER.error("Failed to find value " + moduleName + "/values", e);
            return null;
        }
        
        if(!(values instanceof List))
            return null;
        
        ConcreteModule module = new ConcreteModule(moduleName);
        for(Object item : (List)values) {
            if(!(item instanceof ProceduralValueDefinition))
                return null;
            ProceduralValueDefinition def = (ProceduralValueDefinition)item;

            SCLValue value = new SCLValue(Name.create(moduleName, def.name));
            value.setType(def.type);
            value.setExpression(new EExternalConstant(def.value, def.type));
            value.setInlineInSimplification(true);
            module.addValue(value);
        }
        module.setParentClassLoader(getClass().getClassLoader());
        return new PrecompiledModuleSource(module);
    }

}
