/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.scl.compiler.module.repository;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import gnu.trove.set.hash.THashSet;

/**
 * Listener that is notified about changes in modules and their dependencies. It is possible
 * to listen multiple different modules with one listener. When a change happens, the listener
 * automatically stops listening any other changes. The idea is that the client then asks all modules
 * again using the listener as a parameter.
 */
public abstract class UpdateListener {
    private final THashSet<Observable> observables = new THashSet<Observable>();  
    
    public interface Observable {
        void removeListener(UpdateListener listener);
    }
    
    public abstract void notifyAboutUpdate();
    
    /**
     * Registers an observable to the listener. The client code should never
     * call this method. It is needed so that it is possible to implement
     * {@link #stopListening}.
     */
    public void addObservable(Observable observable) {
        synchronized(observables) {
            observables.add(observable);
        }
    }
    
    public List<Observable> drainObservables() {
        synchronized(observables) {
            if(observables.isEmpty())
                return Collections.emptyList();
            ArrayList<Observable> copy = new ArrayList<>();
            copy.addAll(observables);
            observables.clear();
            return copy;
        }
    }

    /**
     * Stops listening changes. Returns true, if the listener was listening something. 
     */
    public boolean stopListening() {
        // We take a snapshot since calling removeListener with observables monitor can deadlock
        List<Observable> snapshot = drainObservables();
        if(snapshot.isEmpty())
            return false;
        for(Observable observable : snapshot)
            observable.removeListener(this);
        return true;
    }
}
