/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.features;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Session;
import org.simantics.db.common.processor.MergingGraphRequestProcessor;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.project.IProject;
import org.simantics.project.exception.ProjectException;
import org.simantics.utils.datastructures.hints.IHintContext.Key;


/**
 * Implement {@link #configure()} and {@link #deconfigure()} to customize
 * how a feature configures the project in question.
 * 
 * @author Tuukka Lehtonen
 */
public abstract class AbstractProjectFeature implements IProjectFeature {

    private IProject project;

    public AbstractProjectFeature() {
    }

    protected void assertProject() {
        if (project == null)
            throw new IllegalStateException("project element is null");
    }

    @Override
    public IProject getProjectElement() {
        return project;
    }

    public IProject getProject() {
        assertProject();
        return project;
    }

    public IProject peekProject() {
        return project;
    }

    protected Session getSession() {
        IProject p = getProject();
        Session s = p.getSession();
        if (s == null)
            throw new IllegalStateException("project not attached to a database session");
        return s;
    }

    protected Session peekSession() {
        IProject p = peekProject();
        if (p == null)
            return null;
        return p.getSession();
    }

    protected RequestProcessor getGraphRequestProcessor() {
        IProject p = peekProject();
        if (p == null)
            return null;
        Session s = p.getSession();
        //MergingGraphRequestProcessor mgrp = s.getService(MergingGraphRequestProcessor.class);
        MergingGraphRequestProcessor mgrp = null;
        return mgrp != null ? mgrp : s;
    }

    @Override
    public void setProjectElement(IProject project) {
        this.project = project;
    }

    public void onActivated(final ReadGraph graph, final IProject project) throws DatabaseException {
    }

    @Override
    public void configure() throws ProjectException {
        try {
            getSession().syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    onActivated(graph, getProject());
                }
            });
        } catch (DatabaseException e) {
            throw new ProjectException(e);
        }
    }

    @Override
    public void deconfigure() throws ProjectException {
    }

    protected <T> void addToCollectionHint(Key key, T... ts) {
        Collection<T> c = getProjectElement().getHint(key);
        if (c == null) {
            c = new ArrayList<T>();
            getProjectElement().setHint(key, c);
        }
        for (T t : ts)
            c.add(t);
    }

}
