/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project;

import java.util.Set;

import org.simantics.project.features.registry.GroupReference;

/**
 * Utilities for constructing {@link GroupFilter} instances.
 * 
 * @author Tuukka Lehtonen
 */
public final class GroupFilters {

    public static GroupFilter constant(final boolean answer) {
        return new GroupFilter() {
            @Override
            public boolean accept(GroupReference ref) {
                return answer;
            }
        };
    }

    private static final GroupFilter ACCEPT_ALL = constant(true);
    private static final GroupFilter DENY_ALL = constant(false);

    public static GroupFilter acceptAll() {
        return ACCEPT_ALL;
    }

    public static GroupFilter denyAll() {
        return DENY_ALL;
    }

    /**
     * Constructs a group filter that accepts only equal matches of all the
     * group references provided in the argument set.
     * 
     * @param refs the group references to accept
     * @return a filter accepting the specified set of group references
     */
    public static GroupFilter containsEqual(final Set<GroupReference> refs) {
        return new GroupFilter() {
            @Override
            public boolean accept(GroupReference ref) {
                return refs.contains(ref);
            }
        };
    }

    /**
     * Constructs a group filter that accepts groups that include one or
     * more of the specified groups. Inclusion means that:
     * <ol>
     * <li>group ID's are equal</li>
     * <li>if tested group has a version or version range it must include
     * the version in the group of the specified set</li>
     * </ol>
     * 
     * @param refs
     * @return
     */
    public static GroupFilter includesVersion(final Set<GroupReference> refs) {
        return new GroupFilter() {
            @Override
            public boolean accept(GroupReference range) {
                for (GroupReference ref : refs)
                    if (range.includes(ref))
                        return true;
                return false;
            }
        };
    }

}
