/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.management;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.TreeMap;
import java.util.regex.Pattern;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.layer0.DatabaseManagementResource;
import org.simantics.layer0.Layer0;

/**
 * A description of an graph bundle
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class OntologySpec {

	/** Versioned Id pattern */	
	static String ID_PATTERN_STRING =                "[a-zA-Z_0-9]+(?:\\.[a-zA-Z_0-9]+)*";
	static String VERSION_PATTERN_STRING =           "(\\d+).(\\d+).(\\d+).([a-zA-Z_0-9\\-]+)";
	static Pattern ID_PATTERN = Pattern.compile(ID_PATTERN_STRING);
	static Pattern VERSION_PATTERN = Pattern.compile(VERSION_PATTERN_STRING);
	static Pattern VERSIONED_ID_PATTERN = Pattern.compile("(" + ID_PATTERN_STRING + ")/" + VERSION_PATTERN_STRING + "");
	
	/** Display name */
	String name;	
	/** OSGi versioned ID - id/major.minor.micro(.qualifier) */
	String versionedId;
	
	/**
	 * Create new bundle spec 
	 * 
	 * @param name optional name
	 * @param versionedId
	 */
	public OntologySpec(String name, String versionedId) {
		if (!VERSIONED_ID_PATTERN.matcher(versionedId).matches()) 
			throw new IllegalArgumentException("not versioned id");
		this.versionedId = versionedId;
		this.name = name == null ? versionedId : name;
	}
	
	public String getName() {
		return name;
	}

	public String getVersionedId() {
		return versionedId;
	}

	@Override
	public int hashCode() {
		return versionedId.hashCode();
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (obj instanceof OntologySpec == false) return false;
		OntologySpec other = (OntologySpec) obj;
		return other.versionedId.equals(versionedId);
	}
	
	@Override
	public String toString() {
		return name + "("+versionedId+")";
	}

	/** A query that reads all bundles from the database, result is ordered by versioned id */
	static Read<List<OntologySpec>> QUERY = new Read<List<OntologySpec>>() {
		@Override
		public List<OntologySpec> perform(ReadGraph g) throws DatabaseException {
			DatabaseManagementResource DatabaseManagement = DatabaseManagementResource.getInstance(g);
			Layer0 L0 = Layer0.getInstance(g);
			Collection<Resource> tgs = g.getObjects(DatabaseManagement.InstalledGraphBundles, L0.ConsistsOf);
			TreeMap<String, OntologySpec> result = new TreeMap<String, OntologySpec>();			
			for (Resource tg : tgs) {
				if ( !g.isInstanceOf(tg, DatabaseManagement.GraphBundle) ) continue; 
				try {
					String name = g.getPossibleRelatedValue(tg, L0.HasName);
					String versionedId = g.getPossibleRelatedValue(tg, DatabaseManagement.HasVersionedId);
					OntologySpec bs = new OntologySpec(name, versionedId);
					result.put( bs.versionedId, bs );
				} catch (IllegalArgumentException iae) {
					continue;
				}
			}					
			return new ArrayList<OntologySpec>(result.values());
		}
	};
	
}

