/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.project.management;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.function.Supplier;

import org.eclipse.equinox.p2.metadata.IVersionedId;
import org.eclipse.equinox.p2.metadata.Version;
import org.eclipse.equinox.p2.metadata.VersionedId;
import org.simantics.databoard.binding.error.RuntimeBindingException;
import org.simantics.graph.representation.TransferableGraph1;

/**
 * Graph Bundle extended. 
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class GraphBundleEx extends GraphBundle implements IVersionedId {
	
	public static final SimpleDateFormat QUALIFIER_FORMAT = new SimpleDateFormat("yyyyMMddHHmm");
	
	public static GraphBundleEx extend(GraphBundle bundle) {
		return new GraphBundleEx(bundle);
	}
	
	/**
	 * Replace "qualifier" string in version with current time in format of yyyyMMddHHmm. 
	 * 
	 * @param v version
	 * @return v or new version with current time
	 */
	public static Version buildQualifier(Version v) {
		String qualifier = v.getSegment(3).toString();
		if (!qualifier.equals("qualifier")) return v;
		
		int major = (Integer) v.getSegment(0);
		int minor = (Integer) v.getSegment(1);
		int micro = (Integer) v.getSegment(2);		
		qualifier = QUALIFIER_FORMAT.format( new Date() );
		
		return Version.createOSGi(major, minor, micro, qualifier);
	}
	
	public static VersionedId buildQualifier(IVersionedId vid) {
		Version v = vid.getVersion();
		String id = vid.getId();
		return new VersionedId(id, buildQualifier(v));
	}
	
	VersionedId vid;
	
	GraphBundleEx(GraphBundle e) {
		this.graphSource = e.graphSource;
		this.graph = e.graph;
		this.resource = e.resource;
		this.hashcode = e.hashcode;
		this.id = e.id;
		this.major = e.major;
		this.minor = e.minor;
		this.service = e.service;
		this.qualifier = e.qualifier;
		this.resourceArray = e.resourceArray;
		this.name = e.name;
		this.vid = new VersionedId(id, Version.createOSGi(major, minor, service, qualifier));
		this.immutable = e.immutable;
		this.supportMerge = e.supportMerge;
	}
	
	public GraphBundleEx(String name, TransferableGraph1 data, String versionedId) 
	throws RuntimeBindingException 
	{
		super(name, data, versionedId);
		Version v = Version.createOSGi(major, minor, service, qualifier);
		vid = new VersionedId(id, v);
	}	
	
	public GraphBundleEx(String name, TransferableGraph1 data, String id, String version) 
	throws RuntimeBindingException 
	{
		super(name, data, id, version);
		Version v = Version.createOSGi(major, minor, service, qualifier);
		vid = new VersionedId(id, v);
	}	

	public GraphBundleEx(String name, TransferableGraph1 data, String id, Version version) 
	throws RuntimeBindingException 
	{
		super(name, data, id, version.getSegment(0)+"."+version.getSegment(1)+"."+version.getSegment(2)+"."+version.getSegment(3));
		Version v = Version.createOSGi(major, minor, service, qualifier);
		vid = new VersionedId(id, v);
	}	
	
	public GraphBundleEx(String name, TransferableGraph1 data, IVersionedId vid, boolean isImmutable) 
	throws RuntimeBindingException 
	{
		this(name, data, vid, isImmutable, true);
	}

	public GraphBundleEx(String name, TransferableGraph1 data, IVersionedId vid, boolean isImmutable, boolean supportMerge) 
	throws RuntimeBindingException 
	{
		super(name, data, vid.getId(), vid.getVersion().getSegment(0).toString()+"."+vid.getVersion().getSegment(1).toString()+"."+vid.getVersion().getSegment(2).toString()+"."+vid.getVersion().getSegment(3).toString());
		this.vid = new VersionedId(id, vid.getVersion());
		this.immutable = isImmutable;
		this.supportMerge = supportMerge;
	}

	public GraphBundleEx(String name, Supplier<TransferableGraph1> source, int hashValue, IVersionedId vid, boolean isImmutable) 
	throws RuntimeBindingException 
	{
		this(name, source, hashValue, vid, isImmutable, true);
	}

	public GraphBundleEx(String name, Supplier<TransferableGraph1> source, int hashValue, IVersionedId vid, boolean isImmutable, boolean supportMerge) 
	throws RuntimeBindingException 
	{
		super(name, source, hashValue, vid.getId(), vid.getVersion().getSegment(0).toString()+"."+vid.getVersion().getSegment(1).toString()+"."+vid.getVersion().getSegment(2).toString()+"."+vid.getVersion().getSegment(3).toString());
		this.vid = new VersionedId(id, vid.getVersion());
		this.immutable = isImmutable;
		this.supportMerge = supportMerge;
	}

	public GraphBundleEx(String name, TransferableGraph1 data, IVersionedId vid) 
	throws RuntimeBindingException 
	{
		this(name, data, vid, true);
	}
	
	/**
	 * Converts "qualifier" into current time in format of "yyyyMMddHHmm". 
	 */
	public void build() {
		vid = buildQualifier(vid);
		this.qualifier = vid.getVersion().getSegment(3).toString();
	}
	
	public VersionedId getVid() {
		return vid;
	}
	
	public Version getVersion() {
		return vid.getVersion();
	}
	
}

