/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.features;

import org.simantics.project.IProject;
import org.simantics.project.exception.ProjectException;

/**
 * Interface for project feature runtime classes. It can configure a project
 * with the project feature, or de-configure it. When a project is configured
 * with a project feature, this is recorded in the list of project features on
 * the project. Individual project features may expose a more specific runtime
 * type, with additional API for manipulating the project in a feature-specific
 * way.
 * 
 * <p>
 * NOTE: Project features are stateful objects bound to a single
 * <code>IProject</code> (e.g. <code>IProject</code>) at a time. Do not
 * cache or re-use project features.
 * </p>
 * 
 * <p>
 * Clients may implement this interface but extending
 * <code>AbstractProjectFeature</code> is strongly recommended instead.
 * </p>
 * 
 * @see IProject#getFeatures()
 * @see AbstractProjectFeature
 * @see IProjectFeatureDescriptor
 */
public interface IProjectFeature {
    /**
     * Configures this feature for its project. This is called during
     * {@link IProject#activate()} and should not be called directly by clients.
     * 
     * <p>
     * Exceptions thrown by this method will be propagated back to the caller of
     * <code>IProject.activate</code>.
     * 
     * @exception ProjectException if this method fails.
     */
    void configure() throws ProjectException;

    /**
     * De-configures this feature for its project. This is called by the
     * workspace when features are removed from the project using
     * <code>IProject.setDescription</code> and should not be called directly by
     * clients. The feature extension id is removed from the list of features
     * before this method is called, and need not be removed here.
     * 
     * Exceptions thrown by this method will be propagated back to the caller of
     * <code>IProject.deactivate</code>.
     * 
     * @exception ProjectException if this method fails.
     */
    void deconfigure() throws ProjectException;

    /**
     * Returns the project to which this project feature applies. This method
     * may only be invoked from within {@link #configure()} and
     * {@link #deconfigure()}. At other times it is not guaranteed to return the
     * project.
     * 
     * @return the project handle
     */
    IProject getProjectElement();

    /**
     * Sets the project to which this feature applies. This method should not be
     * called directly by clients. It is only to be used internally by
     * {@link IProject#activate()} during project activation and more precisely
     * project feature configuration.
     * 
     * @param project the project to which this feature applies
     */
    void setProjectElement(IProject project);

}
