/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project;

import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.project.exception.ProjectException;
import org.simantics.project.features.IProjectFeature;
import org.simantics.utils.Container;
import org.simantics.utils.datastructures.disposable.IDisposable;
import org.simantics.utils.datastructures.hints.IHintContext;

/**
 * <ul>
 * <li>A project is always related to a single database session.</li>
 * <li>Disposing an IProject shall release all temporarily allocated resources, but it shall not remove anything that will prevent the project from being reloaded</li>
 * <li>Features only configure/deconfigure this IProject by adding/removing things from the IHintContext</li>
 * <li>A feature should only perform a single task</li>
 * <li>Features should be easily composable into new features</li>
 * <li>Life-cycle handlers that perform project-specific (de)initialization tasks, can be added to projects</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 */
public interface IProject extends IDisposable, IHintContext, Container<Resource> {

    /**
     * @return the session this project is related to.
     */
    Session getSession();

    /**
     * @return the features used with this <code>IProject</code>. Depending on
     *         whether the the project is activated or not, the features will be
     *         either configured or unconfigured.
     * @see #activate()
     * @see #deactivate()
     */
    IProjectFeature[] getFeatures();

    /**
     * Activates the project.
     * 
     * <p>
     * This implies invoking all {@link IProjectFeature#configure()} handlers
     * that are attached to this project. The invocation order is the same as in
     * which {@link #getFeatures()} returns the features. If the project has
     * already been activated, this should be a null operation.
     * </p>
     * <p>
     * This method may only be invoked outside of any transaction. This allows
     * the {@link IProjectFeature} handlers to perform arbitrary read and/or
     * write requests into the database.
     * </p>
     */
    void activate() throws ProjectException;

    /**
     * Deactivates the project.
     * 
     * <p>
     * A deactivated project can be activated again later, as long as it isn't
     * disposed.
     * </p>
     * 
     * <p>
     * This implies invoking all {@link IProjectFeature#deconfigure()} handlers
     * that are attached to this project. The invocation order is the reverse of
     * the activation order. If the project is already inactive, this should be
     * a null operation.
     * </p>
     * 
     * <p>
     * Project disposal also implies deactivation.
     * </p>
     */
    void deactivate() throws ProjectException;

}
