/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.typicals;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.Activator;
import org.simantics.scl.runtime.function.Function2;
import org.simantics.ui.workbench.dialogs.ElementListSelectionDialog;
import org.simantics.utils.ui.SWTUtils;

/**
 * @author Tuukka Lehtonen
 */
public class NewTypicalDiagramInstance implements ActionFactory {

    @Override
    public Runnable create(Object target_) {
        Resource target = (Resource) target_;
        return () -> {
            Job job = new DatabaseJob("Instantiate Typical") {
                @Override
                protected IStatus run(IProgressMonitor monitor) {
                    monitor.beginTask("Instantiate Typical...", IProgressMonitor.UNKNOWN);

                    Session session = Simantics.getSession();
                    session.markUndoPoint();
                    try {
                        AtomicReference<Resource> model = new AtomicReference<Resource>();
                        Function2<Resource, Resource, Resource> instantiator = session.syncRequest(new UniqueRead<Function2<Resource, Resource, Resource>>() {
                            @Override
                            public Function2<Resource, Resource, Resource> perform(ReadGraph graph) throws DatabaseException {
                                model.set( graph.sync(new PossibleModel(target)) );
                                if (model.get() == null)
                                    throw new CancelTransactionException("Cannot find a model from the input selection. Typical diagram instantiation not possible.");

                                ModelingResources MOD = ModelingResources.getInstance(graph);
                                Function2<Resource, Resource, Resource> instantiator = graph.getRelatedValue2(model.get(), MOD.StructuralModel_typicalInstantiationFunction);
                                return instantiator;
                            }
                        });

                        instantiator.apply(model.get(), target);

                        return Status.OK_STATUS;
                    } catch (CancelTransactionException e) {
                        return new Status(IStatus.INFO , Activator.PLUGIN_ID, e.getMessage());
                    } catch (DatabaseException e) {
                        return new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e);
                    } finally {
                        monitor.done();
                    }
                }
            };
            //job.setUser(true);
            job.schedule();
        };
    }

    public static <T> void asyncQueryFromList(final Collection<T> elements, final Consumer<T> callback) {
        SWTUtils.asyncExec(PlatformUI.getWorkbench().getDisplay(), () -> {
            T result = queryFromList(elements);
            if (callback != null)
                callback.accept(result);
        });
    }

    @SuppressWarnings("unchecked")
	public static <T> T queryFromList(final Collection<T> elements) {
        IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        Shell shell = window != null ? window.getShell() : null;

        ElementListSelectionDialog dialog = new ElementListSelectionDialog(shell, new LabelProvider());
        dialog.setBlockOnOpen(true);
        dialog.setTitle("Select Template Typical Diagram");
        dialog.setMessage("Select typical diagram to copy (? = any character, * = any string):");
        dialog.setMultipleSelection(false);
        dialog.setMatchEmptyString(true);
        dialog.setElements(elements);
        if (dialog.open() != Window.OK)
            return null;

        return (T) dialog.getFirstResult();
    }

}
