/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.sharedontology.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.util.DraftStatusBean;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.modeling.ModelingUtils.LibraryInfo;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.simantics.utils.ui.ISelectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 */
public class SharedOntologyExportPage extends WizardPage {

    private static final Logger LOGGER = LoggerFactory.getLogger(SharedOntologyExportPage.class);

    ExportPlan          exportModel;
    Composite           draft;
    CCombo              library;
    CCombo              exportLocation;
    private Button      overwrite;

    List<LibraryInfo>   libraries = Collections.emptyList();

    protected SharedOntologyExportPage(ExportPlan model) {
        super("Export Shared Library", "Define Export Location", null);
        this.exportModel = model;
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        draft = new Composite(container, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(draft);
        draft.setBackground(draft.getDisplay().getSystemColor(SWT.COLOR_RED));
        GridLayoutFactory.swtDefaults().spacing(5, 5).applyTo(draft);
        
        Composite draft2 = new Composite(draft, SWT.NONE);
        GridLayoutFactory.swtDefaults().applyTo(draft2);
        GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(draft2);
        new Label(draft2, SWT.NONE).setText("The shared library has not been finished for publishing. The symbol can only be saved with draft status.");
        
        new Label(container, SWT.NONE).setText("Exported &shared library:");
        library = new CCombo(container, SWT.BORDER);
        {
            library.setEditable(false);
            library.setText("");
            library.setToolTipText("Selects the shared library to export.");
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(library);
            library.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    int i = library.getSelectionIndex();
                    exportModel.model = i >= 0 && libraries.size() > 0 ? libraries.get(i) : null;
                    validatePage();
                }
            });
        }

        new Label(container, SWT.NONE).setText("&Target file:");
        exportLocation = new CCombo(container, SWT.BORDER);
        {
            exportLocation.setText("");
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(exportLocation);
            exportLocation.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText("Browse...");
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.SAVE);
                    dialog.setText("Choose Export Target File");
                    String loc = exportLocation.getText();
                    dialog.setFilterPath(loc);
                    dialog.setFilterExtensions(new String[] { "*.sharedLibrary" });
                    dialog.setFilterNames(new String[] { "Shared Library (*.sharedLibrary)" });
                    dialog.setOverwrite(false);
                    String file = dialog.open();
                    if (file == null)
                        return;
                    exportLocation.setText(file);
                    validatePage();
                }
            });
        }

        Label horizRule = new Label(container, SWT.BORDER);
        GridDataFactory.fillDefaults().hint(SWT.DEFAULT, 0).grab(true, false).span(3, 1).applyTo(horizRule);

        overwrite = new Button(container, SWT.CHECK);
        overwrite.setText("&Overwrite existing files without warning");
        overwrite.setSelection(exportModel.overwrite);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(overwrite);
        overwrite.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                validatePage();
            }
        });
        String prop = System.getProperty("org.simantics.modeling.exportTgAndPgraph");
        if (prop != null || Platform.inDevelopmentMode()) {
            Button generateTg = new Button(container, SWT.CHECK);
            generateTg.setText("&Generate transferable graph");
            generateTg.setSelection(exportModel.writeTransferableGraph);
            GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(generateTg);
            generateTg.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
                exportModel.writeTransferableGraph = generateTg.getSelection();
            }));
            Button dumpStructure = new Button(container, SWT.CHECK);
            dumpStructure.setText("&Dump textual shared library structure");
            dumpStructure.setSelection(exportModel.dumpStructure);
            GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(dumpStructure);
            dumpStructure.addSelectionListener(SelectionListener.widgetSelectedAdapter(e -> {
                exportModel.dumpStructure = dumpStructure.getSelection();
            }));
        }

        try {
            initializeData();
        } catch (DatabaseException e) {
            LOGGER.error("Failed to initialize shared ontology wizard export page", e);
        }

        setControl(container);
        validatePage();
    }

    private void initializeData() throws DatabaseException {

        Session session = exportModel.sessionContext.getSession();
        List<Resource> selection = ISelectionUtils.getPossibleKeys(exportModel.selection, SelectionHints.KEY_MAIN, Resource.class);
        AtomicReference<Set<Resource>> selectedRoots = new AtomicReference<>();

        libraries = session.syncRequest(new UniqueRead<List<LibraryInfo>>() {
            @Override
            public List<LibraryInfo> perform(ReadGraph graph) throws DatabaseException {
                List<LibraryInfo> result = new ArrayList<>();

                Set<Resource> roots = toRoots(graph, selection);
                selectedRoots.set(roots);

                List<Resource> libs = Simantics.applySCL("Simantics/SharedOntologies", "getSharedOntologies", graph, Tuple0.INSTANCE);
                for (Resource lib : libs)
                    result.add(library(graph, lib));

                return result;
            }

            private Set<Resource> toRoots(ReadGraph graph, Collection<Resource> resources) throws DatabaseException {
                Set<Resource> result = new HashSet<>();
                for (Resource sel : selection) {
                    Resource root = graph.syncRequest(new PossibleIndexRoot(sel));
                    if (root != null)
                        result.add(root);
                }
                return result;
            }

            private LibraryInfo library(ReadGraph graph, Resource library) throws DatabaseException {
                return new LibraryInfo(
                        NameLabelUtil.modalName(graph, library, NameLabelMode.NAME),
                        library,
                        isDraft(graph, library));
            }

            private DraftStatusBean isDraft(ReadGraph graph, Resource resource) throws DatabaseException {
                return Layer0Utils.isPublished(graph, resource) ? null : new DraftStatusBean(new String[0]);
            }
        });

        for (LibraryInfo lib : libraries) {
            if (selectedRoots.get().contains(lib.library.getResource()))
                exportModel.model = lib;
        }

        // Populate combo boxes
        int i = 0;
        for (LibraryInfo m : libraries) {
            library.add(m.library.getName());
            library.setData(String.valueOf(i), m);
            if (m.equals(exportModel.model))
                library.select(i);
            ++i;
        }

        for (String path : exportModel.recentLocations) {
            exportLocation.add(path);
        }
        if (exportLocation.getItemCount() > 0)
            exportLocation.select(0);
    }

    void validatePage() {
        if (exportModel.model == null) {
            setMessage("Select library to export from.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }

        if(exportModel.model.draft != null) {
            GridLayoutFactory.swtDefaults().spacing(5, 5).applyTo(draft);
            draft.getParent().layout(true);
        } else {
            GridLayoutFactory.swtDefaults().numColumns(0).margins(0, 0).applyTo(draft);
            draft.getParent().layout(true);
        }

        String exportLoc = exportLocation.getText();
        if (exportLoc.isEmpty()) {
            setMessage("Select target file.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        if (!exportLoc.endsWith(".sharedLibrary"))
            exportLoc += exportLoc.endsWith(".") ? "sharedLibrary" : ".sharedLibrary";
        File file = new File(exportLoc);
        if (file.isDirectory()) {
            setErrorMessage("The target is a directory.");
            setPageComplete(false);
            return;
        }
        File parent = file.getParentFile();
        if (parent == null || !parent.isDirectory()) {
            setErrorMessage("The target directory does not exist.");
            setPageComplete(false);
            return;
        }
        exportModel.exportLocation = file;
        exportModel.overwrite = overwrite.getSelection();

        setErrorMessage(null);
        setMessage("Export shared library to " + exportLoc + ".");
        setPageComplete(true);
    }

}
